<?php


namespace Dedi\SyliusTheme\Sales\Form\Type;

use Symfony\Component\Form\AbstractType;
use Sylius\Bundle\AddressingBundle\Form\Type\AddressType as SyliusAddressType;
use Sylius\Bundle\CoreBundle\Form\Type\Checkout\AddressType as BaseAddressType;
use Sylius\Bundle\CoreBundle\Form\Type\Customer\CustomerCheckoutGuestType;
use Sylius\Component\Addressing\Comparator\AddressComparatorInterface;
use Sylius\Component\Core\Model\AddressInterface;
use Sylius\Component\Core\Model\CustomerInterface;
use Sylius\Component\Core\Model\OrderInterface;
use Sylius\Component\Customer\Model\CustomerAwareInterface;
use Symfony\Component\Form\Extension\Core\Type\CheckboxType;
use Symfony\Component\Form\Extension\Core\Type\TextType;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\Form\FormEvent;
use Symfony\Component\Form\FormEvents;
use Symfony\Component\Form\FormTypeInterface;
use Symfony\Component\OptionsResolver\OptionsResolver;
use Symfony\Component\Validator\Constraints\Valid;
use Webmozart\Assert\Assert;

final class AddressType extends AbstractType
{
    private FormTypeInterface $decoratedFormType;

    private AddressComparatorInterface $addressComparator;

    public function __construct(
        BaseAddressType $decoratedFormType,
        AddressComparatorInterface $addressComparator
    ) {
        $this->decoratedFormType = $decoratedFormType;
        $this->addressComparator = $addressComparator;
    }

    public function buildForm(FormBuilderInterface $builder, array $options): void
    {
        $builder
            ->add('differentBillingAddress', CheckboxType::class, [
                'mapped' => false,
                'required' => false,
                'label' => 'sylius.form.checkout.addressing.different_billing_address',
            ])
            ->add('differentShippingAddress', CheckboxType::class, [
                'mapped' => false,
                'required' => false,
                'label' => 'sylius.form.checkout.addressing.different_shipping_address',
            ])
            ->addEventListener(FormEvents::PRE_SET_DATA, static function (FormEvent $event): void {
                $form = $event->getForm();

                Assert::isInstanceOf($event->getData(), OrderInterface::class);

                /** @var OrderInterface $order */
                $order = $event->getData();
                $channel = $order->getChannel();

                $form
                    ->add('shippingAddress', SyliusAddressType::class, [
                        'shippable' => true,
                        'constraints' => [new Valid()],
                        'channel' => $channel,
                    ])
                    ->add('billingAddress', SyliusAddressType::class, [
                        'constraints' => [new Valid()],
                        'channel' => $channel,
                    ])
                ;
            })
            ->addEventListener(FormEvents::POST_SET_DATA, function (FormEvent $event): void {
                $form = $event->getForm();

                Assert::isInstanceOf($event->getData(), OrderInterface::class);

                /** @var OrderInterface $order */
                $order = $event->getData();
                $areAddressesDifferent = $this->areAddressesDifferent($order->getBillingAddress(), $order->getShippingAddress());

                $form->get('differentBillingAddress')->setData($areAddressesDifferent);
                $form->get('differentShippingAddress')->setData($areAddressesDifferent);
            })
            ->addEventListener(FormEvents::PRE_SET_DATA, function (FormEvent $event) use ($options): void {
                $form = $event->getForm();
                $resource = $event->getData();
                $customer = $options['customer'];

                Assert::isInstanceOf($resource, CustomerAwareInterface::class);

                /** @var CustomerInterface|null $resourceCustomer */
                $resourceCustomer = $resource->getCustomer();

                if (
                    (null === $customer && null === $resourceCustomer) ||
                    (null !== $resourceCustomer && null === $resourceCustomer->getUser())
                ) {
                    $form->add('customer', CustomerCheckoutGuestType::class, ['constraints' => [new Valid()]]);
                }
            })
            ->addEventListener(FormEvents::PRE_SUBMIT, function (FormEvent $event): void {
                $orderData = $event->getData();

                $differentBillingAddress = isset($orderData['differentBillingAddress']);
                $differentShippingAddress = isset($orderData['differentShippingAddress']);

                if (!$differentBillingAddress) {
                    $orderData['billingAddress'] = [];
                }

                if ($this->isEmptyAddress($orderData['billingAddress']) && !$differentBillingAddress && !$differentShippingAddress) {
                    $orderData['billingAddress'] = $orderData['shippingAddress'];
                }

                if ($this->isEmptyAddress($orderData['shippingAddress']) && !$differentBillingAddress && !$differentShippingAddress) {
                    $orderData['shippingAddress'] = $orderData['billingAddress'];
                }

                $event->setData($orderData);
            });
    }

    public function configureOptions(OptionsResolver $resolver): void
    {
        $this->decoratedFormType->configureOptions($resolver);
    }

    public function getBlockPrefix(): string
    {
        return $this->decoratedFormType->getBlockPrefix();
    }

    private function isEmptyAddress(array $address): bool
    {
        foreach ($address as $field => $value) {
            if (null !== $value && !empty($value)) {
                return false;
            }
        }

        return true;
    }

    private function areAddressesDifferent(?AddressInterface $firstAddress, ?AddressInterface $secondAddress): bool
    {
        if (null == $this->addressComparator || null == $firstAddress || null == $secondAddress) {
            return false;
        }

        return !$this->addressComparator->equal($firstAddress, $secondAddress);
    }
}