<?php

/*
 * This file is part of the Sylius package.
 *
 * (c) Paweł Jędrzejewski
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

declare(strict_types=1);

namespace Dedi\SyliusTaxonTypePlugin\Form\DataTransformer;

use Sylius\Component\Core\Model\ProductInterface;
use Sylius\Component\Core\Model\ProductTaxonInterface;
use Sylius\Component\Core\Model\TaxonInterface;
use Sylius\Component\Resource\Factory\FactoryInterface;
use Sylius\Component\Resource\Repository\RepositoryInterface;
use Symfony\Component\Form\DataTransformerInterface;
use Symfony\Component\Form\Exception\TransformationFailedException;

final class ProductTaxonToProductTransformer implements DataTransformerInterface
{
    public function __construct(
        private readonly FactoryInterface $productTaxonFactory,
        private readonly RepositoryInterface $productTaxonRepository,
        private readonly TaxonInterface $taxon,
    ) {
    }

    public function transform($value): ?ProductInterface
    {
        if (null === $value) {
            return null;
        }

        $this->assertTransformationValueType($value, ProductTaxonInterface::class);

        return $value->getProduct();
    }

    public function reverseTransform($value): ?ProductTaxonInterface
    {
        if (null === $value) {
            return null;
        }

        $this->assertTransformationValueType($value, ProductInterface::class);

        /** @var ProductTaxonInterface|null $productTaxon */
        $productTaxon = $this->productTaxonRepository->findOneBy(['product' => $value, 'taxon' => $this->taxon]);

        if (null === $productTaxon) {
            /** @var ProductTaxonInterface $productTaxon */
            $productTaxon = $this->productTaxonFactory->createNew();
            $productTaxon->setTaxon($this->taxon);
            $productTaxon->setProduct($value);
        }

        return $productTaxon;
    }

    /**
     * @throws TransformationFailedException
     */
    private function assertTransformationValueType($value, string $expectedType): void
    {
        if (!($value instanceof $expectedType)) {
            throw new TransformationFailedException(
                sprintf(
                    'Expected "%s", but got "%s"',
                    $expectedType,
                    is_object($value) ? get_class($value) : gettype($value),
                ),
            );
        }
    }
}
