<?php

declare(strict_types=1);

namespace Dedi\SyliusShipmentPlugin\Registry;

use Dedi\SyliusShipmentPlugin\Provider\ProviderInterface;
use Sylius\Component\Registry\ExistingServiceException;
use Sylius\Component\Registry\NonExistingServiceException;
use Sylius\Component\Registry\ServiceRegistryInterface;

class ProviderRegistry implements ServiceRegistryInterface
{
    /** @var array<string, object> */
    private array $services = [];

    public function __construct(private string $className, private string $context = 'service')
    {
    }

    public function all(): array
    {
        return $this->services;
    }

    public function register(string $identifier, $service): void
    {
        if ($this->has($identifier)) {
            throw new ExistingServiceException($this->context, $identifier);
        }

        if (!$service instanceof $this->className) {
            throw new \InvalidArgumentException(
                sprintf('%s needs to be of type "%s", "%s" given.', ucfirst($this->context), $this->className, get_class($service)),
            );
        }

        $this->services[$identifier] = $service;
    }

    public function unregister(string $identifier): void
    {
        if (!$this->has($identifier)) {
            throw new NonExistingServiceException($this->context, $identifier, array_keys($this->services));
        }

        unset($this->services[$identifier]);
    }

    public function has(string $identifier): bool
    {
        $hasService = isset($this->services[$identifier]);

        if ($hasService) {
            /** @var ProviderInterface $service */
            $service = $this->services[$identifier];

            return $service->isEnabled();
        }

        return false;
    }

    public function get(string $identifier): object
    {
        if (!$this->has($identifier)) {
            throw new NonExistingServiceException($this->context, $identifier, array_keys($this->services));
        }

        return $this->services[$identifier];
    }
}
