<?php

declare(strict_types=1);

namespace Dedi\SyliusShipmentPlugin\Provider\Tnt;

use Dedi\Contracts\SyliusPluginSettings\SettingsProviderInterface;
use Dedi\SyliusShipmentPlugin\Client\ClientInterface;
use Dedi\SyliusShipmentPlugin\DependencyInjection\Settings\SettingsInterface;
use Dedi\SyliusShipmentPlugin\Entity\PickupPointCode;
use Dedi\SyliusShipmentPlugin\Entity\PickupPointInterface;
use Dedi\SyliusShipmentPlugin\Formater\OpeningHoursFormaterInterface;
use Dedi\SyliusShipmentPlugin\Provider\Provider;
use Psr\Log\LoggerInterface;
use Sylius\Component\Core\Model\OrderInterface;
use Sylius\Component\Order\Context\CartContextInterface;
use Sylius\Component\Resource\Factory\FactoryInterface;
use Webmozart\Assert\Assert;

class TntProvider extends Provider
{
    public function __construct(
        private ClientInterface $client,
        private OpeningHoursFormaterInterface $openingHoursFormater,
        private FactoryInterface $pickupPointFactory,
        private SettingsProviderInterface $settingsProvider,
        private CartContextInterface $cartContext,
        private LoggerInterface $logger,
    ) {
    }

    public function getCode(): string
    {
        return 'tnt';
    }

    public function getName(): string
    {
        return 'Tnt';
    }

    public function findPickupPoints(OrderInterface $order): iterable
    {
        $address = $order->getShippingAddress();
        Assert::notNull($address);

        $countryCode = $address->getCountryCode();
        Assert::notNull($countryCode);

        $postCode = $address->getPostcode();
        Assert::notNull($postCode);

        $city = $address->getCity();
        Assert::notNull($city);

        $street = $address->getStreet();
        Assert::notNull($street);

        try {
            $response = $this->client->call('dropOffPoints', [
                'zipCode' => $postCode,
                'city' => $city,
            ]);

            $response = \json_decode(\json_encode($response), true);
        } catch (\Exception $e) {
            $this->logger->error($e->getMessage());

            return [];
        }

        $pickupPoints = [];

        foreach ($response['DropOffPoint'] as $item) {
            $pickupPoints[] = $this->createPickupPoint($item);
        }

        return $pickupPoints;
    }

    public function findPickupPoint(PickupPointCode $code): ?PickupPointInterface
    {
        $cart = $this->cartContext->getCart();

        Assert::isInstanceOf($cart, OrderInterface::class);

        try {
            $points = $this->findPickupPoints($cart);
            $point = null;
            foreach ($points as $pickupPoint) {
                if ($pickupPoint->getCode()->getIdPart() === $code->getIdPart()) {
                    $point = $pickupPoint;

                    break;
                }
            }
            if (null === $point) {
                throw new \Exception('Pickup point not found');
            }

            return $point;
        } catch (\Exception $e) {
            $this->logger->error($e->getMessage());

            return null;
        }
    }

    public function findAllPickupPoints(): iterable
    {
        return [];
    }

    private function createPickupPoint(array $data): PickupPointInterface
    {
        /** @var PickupPointInterface $pickupPoint */
        $pickupPoint = $this->pickupPointFactory->createNew();

        $pickupPoint->setCode(new PickupPointCode($data['xETTCode'], $this->getCode(), 'FR'));
        $pickupPoint->setName($data['name']);
        $pickupPoint->setAddress($data['address1']);
        $pickupPoint->setZipCode($data['zipCode']);
        $pickupPoint->setCity($data['city']);
        $pickupPoint->setCountry('FR');
        $pickupPoint->setLongitude((string) $data['longitude']);
        $pickupPoint->setLatitude((string) $data['latitude']);
        $pickupPoint->setOpeningHours($this->openingHoursFormater->format($data));

        return $pickupPoint;
    }

    public function isEnabled(): bool
    {
        $settings = $this->settingsProvider->provide();

        Assert::isInstanceOf($settings, SettingsInterface::class);

        return $settings->isTntEnabled();
    }
}
