<?php

declare(strict_types=1);

namespace Dedi\SyliusShipmentPlugin\Provider\Gls;

use Dedi\Contracts\SyliusPluginSettings\SettingsProviderInterface;
use Dedi\SyliusShipmentPlugin\Client\ClientInterface;
use Dedi\SyliusShipmentPlugin\DependencyInjection\Settings\SettingsInterface;
use Dedi\SyliusShipmentPlugin\Entity\PickupPointCode;
use Dedi\SyliusShipmentPlugin\Entity\PickupPointInterface;
use Dedi\SyliusShipmentPlugin\Formater\OpeningHoursFormaterInterface;
use Dedi\SyliusShipmentPlugin\Provider\Provider;
use function preg_replace;
use Psr\Log\LoggerInterface;
use Sylius\Component\Core\Model\OrderInterface;
use Sylius\Component\Resource\Factory\FactoryInterface;
use Webmozart\Assert\Assert;

final class GlsProvider extends Provider
{
    public function __construct(
        private ClientInterface $client,
        private FactoryInterface $pickupPointFactory,
        private SettingsProviderInterface $settingsProvider,
        private OpeningHoursFormaterInterface $openingHoursFormater,
        private LoggerInterface $logger,
        private array $countryCodes = ['FR'],
    ) {
    }

    public function findPickupPoints(OrderInterface $order): iterable
    {
        $shippingAddress = $order->getShippingAddress();
        if (null === $shippingAddress) {
            return [];
        }

        $street = $shippingAddress->getStreet();
        $postCode = $shippingAddress->getPostcode();
        $countryCode = $shippingAddress->getCountryCode();

        if (null === $street || null === $postCode || null === $countryCode) {
            return [];
        }

        try {
            $response = $this->client->call('SearchNearestParcelShops', [
                'street' => $street,
                'zipcode' => preg_replace('/\s+/', '', $postCode),
                'countryIso3166A2' => $countryCode,
                'Amount' => 10,
            ]);

            $response = \json_decode(\json_encode($response), true);
        } catch (\Exception $e) {
            $this->logger->error($e->getMessage());

            return [];
        }

        $pickupPoints = [];
        foreach ($response['SearchNearestParcelShopsResult']['parcelshops']['PakkeshopData'] as $item) {
            $pickupPoints[] = $this->createPickupPoint($item);
        }

        return $pickupPoints;
    }

    public function findPickupPoint(PickupPointCode $code): ?PickupPointInterface
    {
        try {
            $response = $this->client->call('GetOneParcelShop', [
                'ParcelShopNumber' => $code->getIdPart(),
            ]);

            $response = \json_decode(\json_encode($response), true);
        } catch (\Exception $e) {
            $this->logger->error($e->getMessage());

            return null;
        }

        return $this->createPickupPoint($response['GetOneParcelShopResult']);
    }

    public function findAllPickupPoints(): iterable
    {
        foreach ($this->countryCodes as $countryCode) {
            try {
                $response = $this->client->call('GetAllParcelShops', [
                    'countryIso3166A2' => $countryCode,
                ]);

                $response = \json_decode(\json_encode($response), true);

                foreach ($response['GetAllParcelShopsResult']['PakkeshopData'] as $item) {
                    yield $this->createPickupPoint($item);
                }
            } catch (\Exception $e) {
                $this->logger->error($e->getMessage());

                continue;
            }
        }
    }

    public function getCode(): string
    {
        return 'gls';
    }

    public function getName(): string
    {
        return 'GLS';
    }

    private function createPickupPoint(array $data): PickupPointInterface
    {
        /** @var PickupPointInterface|object $pickupPoint */
        $pickupPoint = $this->pickupPointFactory->createNew();

        Assert::isInstanceOf($pickupPoint, PickupPointInterface::class);

        $pickupPoint->setCode(new PickupPointCode($data['Number'], $this->getCode(), $data['CountryCodeISO3166A2']));
        $pickupPoint->setName($data['CompanyName']);
        $pickupPoint->setAddress($data['Streetname']);
        $pickupPoint->setAddress2($data['Streetname2']);
        $pickupPoint->setZipCode($data['ZipCode']);
        $pickupPoint->setCity($data['CityName']);
        $pickupPoint->setCountry($data['CountryCodeISO3166A2']);
        $pickupPoint->setLatitude($data['Latitude']);
        $pickupPoint->setLongitude($data['Longitude']);
        $pickupPoint->setOpeningHours($this->openingHoursFormater->format($data));

        return $pickupPoint;
    }

    public function isEnabled(): bool
    {
        $settings = $this->settingsProvider->provide();

        Assert::isInstanceOf($settings, SettingsInterface::class);

        return $settings->isGlsEnabled();
    }
}
