<?php

declare(strict_types=1);

namespace Dedi\SyliusShipmentPlugin\Provider;

use Dedi\SyliusShipmentPlugin\Entity\PickupPointCode;
use Dedi\SyliusShipmentPlugin\Entity\PickupPointInterface;
use Faker\Factory;
use Faker\Generator;
use Sylius\Component\Core\Model\OrderInterface;
use Sylius\Component\Resource\Factory\FactoryInterface;
use Webmozart\Assert\Assert;

final class FakerProvider extends Provider
{
    private Generator $faker;

    public function __construct(private FactoryInterface $pickupPointFactory)
    {
        $this->faker = Factory::create();
    }

    public function findPickupPoints(OrderInterface $order): iterable
    {
        $address = $order->getShippingAddress();
        Assert::notNull($address);

        $countryCode = $address->getCountryCode();
        Assert::notNull($countryCode);

        $pickupPoints = [];
        for ($i = 0; $i < 10; ++$i) {
            $pickupPoints[] = $this->createFakePickupPoint((string) $i, $countryCode);
        }

        return $pickupPoints;
    }

    public function findPickupPoint(PickupPointCode $code): ?PickupPointInterface
    {
        return $this->createFakePickupPoint($code->getIdPart(), $code->getCountryPart());
    }

    public function findAllPickupPoints(): iterable
    {
        for ($i = 0; $i < 10; ++$i) {
            yield $this->createFakePickupPoint((string) $i);
        }
    }

    public function getCode(): string
    {
        return 'faker';
    }

    public function getName(): string
    {
        return 'Faker';
    }

    private function createFakePickupPoint(string $index, ?string $countryCode = null): PickupPointInterface
    {
        if (null === $countryCode) {
            $countryCode = $this->faker->countryCode;
        }

        /** @var PickupPointInterface|object $pickupPoint */
        $pickupPoint = $this->pickupPointFactory->createNew();

        Assert::isInstanceOf($pickupPoint, PickupPointInterface::class);

        $pickupPoint->setCode(new PickupPointCode($index, $this->getCode(), $countryCode));
        $pickupPoint->setName("Post office #$index");
        $pickupPoint->setAddress($this->faker->streetAddress);
        $pickupPoint->setZipCode((string) $this->faker->numberBetween(11111, 99999));
        $pickupPoint->setCity($this->faker->city);
        $pickupPoint->setCountry($countryCode);
        $pickupPoint->setLatitude((string) $this->faker->latitude);
        $pickupPoint->setLongitude((string) $this->faker->longitude);

        return $pickupPoint;
    }

    public function isEnabled(): bool
    {
        return true;
    }
}
