<?php

declare(strict_types=1);

namespace Dedi\SyliusShipmentPlugin\Provider\Dpd;

use Dedi\Contracts\SyliusPluginSettings\SettingsProviderInterface;
use Dedi\SyliusShipmentPlugin\Client\ClientInterface;
use Dedi\SyliusShipmentPlugin\DependencyInjection\Settings\SettingsInterface;
use Dedi\SyliusShipmentPlugin\Entity\PickupPointCode;
use Dedi\SyliusShipmentPlugin\Entity\PickupPointInterface;
use Dedi\SyliusShipmentPlugin\Formater\OpeningHoursFormaterInterface;
use Dedi\SyliusShipmentPlugin\Provider\Provider;
use Psr\Log\LoggerInterface;
use SimpleXMLElement;
use Sylius\Component\Core\Model\OrderInterface;
use Sylius\Component\Resource\Factory\FactoryInterface;
use Webmozart\Assert\Assert;

class DpdProvider extends Provider
{
    public function __construct(
        private ClientInterface $client,
        private OpeningHoursFormaterInterface $openingHoursFormater,
        private FactoryInterface $pickupPointFactory,
        private SettingsProviderInterface $settingsProvider,
        private LoggerInterface $logger,
    ) {
    }

    public function getCode(): string
    {
        return 'dpd';
    }

    public function getName(): string
    {
        return 'Dpd';
    }

    public function findPickupPoints(OrderInterface $order): iterable
    {
        $address = $order->getShippingAddress();
        Assert::notNull($address);

        $countryCode = $address->getCountryCode();
        Assert::notNull($countryCode);

        $postCode = $address->getPostcode();
        Assert::notNull($postCode);

        $city = $address->getCity();
        Assert::notNull($city);

        $street = $address->getStreet();
        Assert::notNull($street);

        try {
            $response = $this->client->call('GetPudoList', [
                'address' => $street,
                'zipCode' => $postCode,
                'city' => $city,
                'countryCode' => $countryCode,
                'requestID' => $order->getId(),
                'date_from' => (new \DateTime())->format('d/m/Y'),
                'max_pudo_number' => 10,
                'max_distance_search' => '',
                'weight' => '',
                'category' => '',
                'holiday_tolerant' => '',
            ]);

            $data = new SimpleXMLElement($response->getBody()->getContents());
        } catch (\Exception $e) {
            $this->logger->error($e->getMessage());

            return [];
        }

        $pickupPoints = [];

        foreach ($data->PUDO_ITEMS->PUDO_ITEM as $item) {
            $pickupPoints[] = $this->createPickupPoint((array) $item, $countryCode);
        }

        return $pickupPoints;
    }

    public function findPickupPoint(PickupPointCode $code): ?PickupPointInterface
    {
        try {
            $response = $this->client->call('GetPudoDetails', [
                'pudo_id' => $code->getIdPart(),
            ]);

            $data = new SimpleXMLElement($response->getBody()->getContents());
        } catch (\Exception $e) {
            $this->logger->error($e->getMessage());

            return null;
        }

        return $this->createPickupPoint((array) $data->PUDO_ITEMS->PUDO_ITEM, $code->getCountryPart());
    }

    public function findAllPickupPoints(): iterable
    {
        return [];
    }

    private function createPickupPoint(array $data, string $countryCode): PickupPointInterface
    {
        /** @var PickupPointInterface $pickupPoint */
        $pickupPoint = $this->pickupPointFactory->createNew();

        $pickupPoint->setCode(new PickupPointCode($data['PUDO_ID'], $this->getCode(), $countryCode));
        $pickupPoint->setName($data['NAME']);
        $pickupPoint->setAddress($data['ADDRESS1']);
        if (isset($data['ADDRESS2']) && is_string($data['ADDRESS2'])) {
            $pickupPoint->setAddress2($data['ADDRESS2']);
        }
        if (isset($data['ADDRESS3']) && is_string($data['ADDRESS3'])) {
            $pickupPoint->setAddress3($data['ADDRESS3']);
        }
        $pickupPoint->setZipCode($data['ZIPCODE']);
        $pickupPoint->setCity($data['CITY']);
        $pickupPoint->setCountry($countryCode);
        $pickupPoint->setLongitude(strtr($data['LONGITUDE'], ',', '.'));
        $pickupPoint->setLatitude(strtr($data['LATITUDE'], ',', '.'));
        $pickupPoint->setOpeningHours($this->openingHoursFormater->format($data));

        return $pickupPoint;
    }

    public function isEnabled(): bool
    {
        $settings = $this->settingsProvider->provide();

        Assert::isInstanceOf($settings, SettingsInterface::class);

        return $settings->isDpdEnabled();
    }
}
