<?php

declare(strict_types=1);

namespace Dedi\SyliusShipmentPlugin\Provider\Colissimo;

use Dedi\Contracts\SyliusPluginSettings\SettingsProviderInterface;
use Dedi\SyliusShipmentPlugin\Client\ClientInterface;
use Dedi\SyliusShipmentPlugin\DependencyInjection\Settings\SettingsInterface;
use Dedi\SyliusShipmentPlugin\Entity\PickupPointCode;
use Dedi\SyliusShipmentPlugin\Entity\PickupPointInterface;
use Dedi\SyliusShipmentPlugin\Formater\OpeningHoursFormaterInterface;
use Dedi\SyliusShipmentPlugin\Provider\Provider;
use Psr\Log\LoggerInterface;
use Sylius\Component\Core\Model\OrderInterface;
use Sylius\Component\Resource\Factory\FactoryInterface;
use Webmozart\Assert\Assert;

class ColissimoProvider extends Provider
{
    public function __construct(
        private ClientInterface $client,
        private OpeningHoursFormaterInterface $openingHoursFormater,
        private FactoryInterface $pickupPointFactory,
        private SettingsProviderInterface $settingsProvider,
        private LoggerInterface $logger,
    ) {
    }

    public function getCode(): string
    {
        return 'colissimo';
    }

    public function getName(): string
    {
        return 'Colissimo';
    }

    public function findPickupPoints(OrderInterface $order): iterable
    {
        $address = $order->getShippingAddress();
        Assert::notNull($address);

        $countryCode = $address->getCountryCode();
        Assert::notNull($countryCode);

        $postCode = $address->getPostcode();
        Assert::notNull($postCode);

        $city = $address->getCity();
        Assert::notNull($city);

        try {
            $response = $this->client->call('findRDVPointRetraitAcheminement', [
                'zipCode' => $postCode,
                'city' => $city,
                'weight' => 1,
                'countryCode' => $countryCode,
                'shippingDate' => (new \DateTime())->format('d/m/Y'),
                'optionInter' => 1,
                'pickupPointType' => 'all',
            ]);

            $response = \json_decode(\json_encode($response), true);
            $this->validateResponse($response);
        } catch (\Exception $e) {
            $this->logger->error($e->getMessage());

            return [];
        }

        $pickupPoints = [];

        foreach ($response['return']['listePointRetraitAcheminement'] as $pointRetrait) {
            $pickupPoints[] = $this->createPickupPoint($pointRetrait);
        }

        return $pickupPoints;
    }

    public function findPickupPoint(PickupPointCode $code): ?PickupPointInterface
    {
        try {
            $response = $this->client->call('findPointRetraitAcheminementByID', [
                'id' => $code->getIdPart(),
                'weight' => 1,
                'date' => (new \DateTime())->format('d/m/Y'),
                'filterRelay' => 0,
            ]);

            $response = \json_decode(\json_encode($response), true);
            $this->validateResponse($response);
        } catch (\Exception $e) {
            $this->logger->error($e->getMessage());

            return null;
        }

        $pickupPoint = $response['return']['pointRetraitAcheminement'];

        return $this->createPickupPoint($pickupPoint);
    }

    public function findAllPickupPoints(): iterable
    {
        return [];
    }

    private function validateResponse(array $response): void
    {
        if (!isset($response['return']['errorCode']) || 0 === (int) $response['return']['errorCode']) {
            return;
        }

        $error = 'Unexpected error';
        if (isset($response['return']['errorMessage']) && '' !== $response['return']['errorMessage']) {
            $error = $response['return']['errorMessage'];
        }

        throw new \RuntimeException($error, (int) $response['return']['errorCode']);
    }

    private function createPickupPoint(array $data): PickupPointInterface
    {
        /** @var PickupPointInterface $pickupPoint */
        $pickupPoint = $this->pickupPointFactory->createNew();

        $pickupPoint->setCode(new PickupPointCode($data['identifiant'], $this->getCode(), $data['codePays']));
        $pickupPoint->setName($data['nom']);
        $pickupPoint->setAddress($data['adresse1']);
        $pickupPoint->setAddress2($data['adresse2']);
        $pickupPoint->setAddress3($data['adresse3']);
        $pickupPoint->setZipCode($data['codePostal']);
        $pickupPoint->setCity($data['localite']);
        $pickupPoint->setCountry($data['codePays']);
        $pickupPoint->setLongitude($data['coordGeolocalisationLongitude']);
        $pickupPoint->setLatitude($data['coordGeolocalisationLatitude']);
        $pickupPoint->setOpeningHours($this->openingHoursFormater->format($data));

        return $pickupPoint;
    }

    public function isEnabled(): bool
    {
        $settings = $this->settingsProvider->provide();

        Assert::isInstanceOf($settings, SettingsInterface::class);

        return $settings->isColissimoEnabled();
    }
}
