<?php

declare(strict_types=1);

namespace Dedi\SyliusShipmentPlugin\Modifier;

use Dedi\SyliusShipmentPlugin\Entity\PickupPointAwareInterface;
use Dedi\SyliusShipmentPlugin\Entity\PickupPointInterface;
use Dedi\SyliusShipmentPlugin\Entity\ShipmentInterface;
use Dedi\SyliusShipmentPlugin\Entity\TempAddressAwareInterface;
use Doctrine\ORM\EntityManagerInterface;
use Sylius\Component\Core\Model\AddressInterface;
use Sylius\Component\Core\Model\OrderInterface;
use Sylius\Component\Resource\Factory\FactoryInterface;
use Symfony\Component\Form\DataTransformerInterface;
use Webmozart\Assert\Assert;

class OrderAddressModifier implements OrderAddressModifierInterface
{
    /**
     * @param DataTransformerInterface<PickupPointInterface|null, string|null> $transformer
     */
    public function __construct(
        private DataTransformerInterface $transformer,
        private FactoryInterface $addressFactory,
        private EntityManagerInterface $entityManager,
    ) {
    }

    public function updateOrderShippingAddress(OrderInterface $order): void
    {
        Assert::isInstanceOf($order, TempAddressAwareInterface::class);

        $shipment = $order->getShipments()->first();

        if (!$shipment instanceof ShipmentInterface) {
            return;
        }

        if ($shipment->hasPickupPointId()) {
            if (null === $order->getTempShippingAddress()) {
                $this->storeTempOrderShippingAddress($order);
            }
            $this->updateShippingAddressForProvider($order, $shipment);
        } elseif (null !== $order->getTempShippingAddress()) {
            $shippingAddress = $order->getShippingAddress();
            if (null !== $shippingAddress) {
                $this->entityManager->remove($shippingAddress);
            }
            $order->setShippingAddress($order->getTempShippingAddress());
            $order->setTempShippingAddress(null);
        }
    }

    public function removeTempAddress(OrderInterface $order): void
    {
        Assert::isInstanceOf($order, TempAddressAwareInterface::class);

        if (null !== $order->getTempShippingAddress()) {
            $this->entityManager->remove($order->getTempShippingAddress());
            $order->setTempShippingAddress(null);
        }
    }

    private function updateShippingAddressForProvider(OrderInterface $order, ShipmentInterface $shipment): void
    {
        $pickupPoint = $this->transformer->reverseTransform($shipment->getPickupPointId());

        if (!$pickupPoint instanceof PickupPointInterface) {
            return;
        }

        Assert::isInstanceOf($order, TempAddressAwareInterface::class);

        $tempShippingAddress = $order->getTempShippingAddress();
        $shippingAddress = $order->getShippingAddress();

        /** @var AddressInterface|PickupPointAwareInterface $newAddress */
        $newAddress = $this->addressFactory->createNew();

        $newAddress->setCompany($pickupPoint->getName());
        $newAddress->setStreet($pickupPoint->getAddress());
        $newAddress->setPostcode($pickupPoint->getZipCode());
        $newAddress->setCity($pickupPoint->getCity());
        $newAddress->setCountryCode($pickupPoint->getCountry());

        $targetAddress = $tempShippingAddress instanceof AddressInterface ? $tempShippingAddress : $shippingAddress;

        if (null !== $targetAddress) {
            $newAddress->setPhoneNumber($targetAddress->getPhoneNumber());
            $newAddress->setFirstName($targetAddress->getFirstName());
            $newAddress->setLastName($targetAddress->getLastName());
            $newAddress->setProvinceCode($targetAddress->getProvinceCode());
            $newAddress->setProvinceName($targetAddress->getProvinceName());

            $order->setShippingAddress($newAddress);
        }
    }

    private function storeTempOrderShippingAddress(OrderInterface $order): void
    {
        Assert::isInstanceOf($order, TempAddressAwareInterface::class);

        $shipment = $order->getShipments()->first();

        if (!$shipment instanceof ShipmentInterface) {
            return;
        }

        if (!$shipment->hasPickupPointId()) {
            return;
        }

        if (null === $order->getTempShippingAddress()) {
            $order->setTempShippingAddress($order->getShippingAddress());
        }
    }
}
