<?php

declare(strict_types=1);

namespace Dedi\SyliusShipmentPlugin\Form\Extension;

use Dedi\SyliusShipmentPlugin\Entity\PickupPointProviderAwareInterface;
use Dedi\SyliusShipmentPlugin\Provider\ProviderInterface;
use Sylius\Bundle\ShippingBundle\Form\Type\ShippingMethodChoiceType;
use Sylius\Component\Core\Model\ShippingMethodInterface;
use Sylius\Component\Core\Repository\ShippingMethodRepositoryInterface;
use Sylius\Component\Order\Context\CartContextInterface;
use Sylius\Component\Registry\ServiceRegistryInterface;
use Symfony\Component\Form\AbstractTypeExtension;
use Symfony\Component\OptionsResolver\OptionsResolver;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Symfony\Component\Security\Csrf\CsrfTokenManagerInterface;
use Webmozart\Assert\Assert;

final class ShippingMethodChoiceTypeExtension extends AbstractTypeExtension
{
    /**
     * @param ShippingMethodRepositoryInterface<ShippingMethodInterface> $shippingMethodRepository
     */
    public function __construct(
        private ServiceRegistryInterface $providerRegistry,
        private CartContextInterface $cartContext,
        private CsrfTokenManagerInterface $csrfTokenManager,
        private ShippingMethodRepositoryInterface $shippingMethodRepository,
        private UrlGeneratorInterface $urlGenerator,
    ) {
    }

    public function configureOptions(OptionsResolver $resolver): void
    {
        $defaultAttr = [
            'class' => 'input-shipping-method',
        ];

        $resolver->setDefault('choice_attr', function (mixed $choiceValue, $key, $value) use ($defaultAttr): array {
            if (is_string($choiceValue)) {
                $choiceValue = $this->shippingMethodRepository->findOneBy(['code' => $choiceValue]);
            }

            Assert::isInstanceOf($choiceValue, PickupPointProviderAwareInterface::class);
            Assert::isInstanceOf($choiceValue, ShippingMethodInterface::class);

            $defaultAttr['data-summary-url'] = $this->urlGenerator->generate(
                'dedi_sylius_shipment_plugin_order_summary_updater',
                ['code' => $choiceValue->getCode()],
            );

            if (!$choiceValue->hasPickupPointProvider()) {
                return $defaultAttr;
            }

            /** @var string $pickupPointProviderId */
            $pickupPointProviderId = $choiceValue->getPickupPointProvider();
            if (!$this->providerRegistry->has($pickupPointProviderId)) {
                return $defaultAttr;
            }

            /** @var ProviderInterface $provider */
            $provider = $this->providerRegistry->get($pickupPointProviderId);

            return [
                    'data-pickup-point-provider' => $provider->getCode(),
                    'data-csrf-token' => $this->csrfTokenManager->getToken((string) $this->cartContext->getCart()->getId()),
                ] + $defaultAttr;
        });
    }

    public static function getExtendedTypes(): iterable
    {
        return [ShippingMethodChoiceType::class];
    }
}
