<?php

declare(strict_types=1);

namespace Dedi\SyliusShipmentPlugin\Form\DataTransformer;

use Dedi\SyliusShipmentPlugin\Entity\PickupPointCode;
use Dedi\SyliusShipmentPlugin\Entity\PickupPointInterface;
use Dedi\SyliusShipmentPlugin\Provider\ProviderInterface;
use function sprintf;
use Sylius\Component\Registry\ServiceRegistryInterface;
use Symfony\Component\Form\DataTransformerInterface;
use Symfony\Component\Form\Exception\TransformationFailedException;

/**
 * @implements DataTransformerInterface<PickupPointInterface|null, PickupPointCode|string|null>
 */
final class PickupPointToIdentifierTransformer implements DataTransformerInterface
{
    public function __construct(private ServiceRegistryInterface $providerRegistry)
    {
    }

    /**
     * @param mixed|PickupPointInterface $value
     */
    public function transform($value): ?PickupPointCode
    {
        if (null === $value) {
            return null;
        }

        $this->assertTransformationValueType($value, PickupPointInterface::class);

        return $value->getCode();
    }

    /**
     * @param mixed $value
     */
    public function reverseTransform($value): ?PickupPointInterface
    {
        if (null === $value) {
            return null;
        }

        $pickupPointId = PickupPointCode::createFromString($value);

        /** @var ProviderInterface $provider */
        $provider = $this->providerRegistry->get($pickupPointId->getProviderPart());

        /** @var PickupPointInterface $pickupPoint */
        $pickupPoint = $provider->findPickupPoint($pickupPointId);

        $this->assertTransformationValueType($pickupPoint, PickupPointInterface::class);

        return $pickupPoint;
    }

    /**
     * @template ExpectedType of object
     *
     * @param class-string<ExpectedType> $expectedType
     *
     * @psalm-assert ExpectedType $value
     */
    private function assertTransformationValueType(mixed $value, string $expectedType): void
    {
        if (!$value instanceof $expectedType) {
            throw new TransformationFailedException(
                sprintf(
                    'Expected "%s", but got "%s"',
                    $expectedType,
                    get_debug_type($value),
                ),
            );
        }
    }
}
