<?php

declare(strict_types=1);

namespace Dedi\SyliusShipmentPlugin\Fixture\Factory;

use Dedi\SyliusShipmentPlugin\Entity\PickupPointCode;
use Dedi\SyliusShipmentPlugin\Entity\PickupPointInterface;
use Dedi\SyliusShipmentPlugin\Provider\ProviderInterface;
use Dedi\SyliusShipmentPlugin\Registry\ProviderRegistry;
use Faker\Factory;
use Faker\Generator;
use Sylius\Bundle\CoreBundle\Fixture\Factory\AbstractExampleFactory;
use Sylius\Bundle\CoreBundle\Fixture\Factory\ExampleFactoryInterface;
use Sylius\Component\Resource\Factory\FactoryInterface;
use Symfony\Component\OptionsResolver\Options;
use Symfony\Component\OptionsResolver\OptionsResolver;

class PickupPointExampleFactory extends AbstractExampleFactory implements ExampleFactoryInterface
{
    private Generator $faker;

    private OptionsResolver $optionsResolver;

    public function __construct(
        private FactoryInterface $pickupPointFactory,
        private ProviderRegistry $providerRegistry,
    ) {
        $this->optionsResolver = new OptionsResolver();
        $this->faker = Factory::create();

        $this->configureOptions($this->optionsResolver);
    }

    public function create(array $options = []): PickupPointInterface
    {
        $options = $this->optionsResolver->resolve($options);

        /** @var PickupPointInterface $pickupPoint */
        $pickupPoint = $this->pickupPointFactory->createNew();

        $pickupPointCode = new PickupPointCode(
            $options['code']['id'],
            $options['code']['provider'],
            $options['code']['country'],
        );

        $pickupPoint->setName($options['name']);
        $pickupPoint->setAddress($options['address']);
        $pickupPoint->setZipCode($options['zip_code']);
        $pickupPoint->setCity($options['city']);
        $pickupPoint->setCountry($options['country']);
        $pickupPoint->setCode($pickupPointCode);
        $pickupPoint->setOpeningHours($options['opening_hours']);

        if (array_key_exists('address2', $options) && null !== $options['address2']) {
            $pickupPoint->setAddress2($options['address2']);
        }
        if (array_key_exists('address3', $options) && null !== $options['address3']) {
            $pickupPoint->setAddress3($options['address3']);
        }
        if (array_key_exists('latitude', $options)) {
            $pickupPoint->setLatitude($options['latitude']);
        }
        if (array_key_exists('longitude', $options)) {
            $pickupPoint->setLongitude($options['longitude']);
        }

        return $pickupPoint;
    }

    protected function configureOptions(OptionsResolver $resolver): void
    {
        $countryCode = $this->faker->countryCode;

        $resolver
            ->setDefault('name', function (Options $options): string {
                /** @var string $words */
                $words = $this->faker->words(3, true);

                return $words;
            })
            ->setDefault('address', function (Options $options): string {
                /** @var string $address */
                $address = $this->faker->streetAddress;

                return $address;
            })
            ->setDefault('zip_code', function (Options $options): string {
                /** @var string $zipCode */
                $zipCode = $this->faker->postcode;

                return $zipCode;
            })
            ->setDefault('city', function (Options $options): string {
                /** @var string $city */
                $city = $this->faker->city;

                return $city;
            })
            ->setDefault('country', function (Options $options) use ($countryCode): string {
                return $countryCode;
            })
            ->setDefault('code', function (Options $options) use ($countryCode): array {
                /** @var ProviderInterface[] $providers */
                $providers = $this->providerRegistry->all();
                $providers = array_filter($providers, fn (ProviderInterface $provider) => !in_array($provider->getCode(), ['store', 'faker'], true));

                $provider = $this->faker->randomElement($providers);

                return [
                    'id' => $this->faker->uuid,
                    'provider' => $provider->getCode(),
                    'country' => $countryCode,
                ];
            })
            ->setDefault('opening_hours', function (Options $options): array {
                return [];
            })
            ->setDefined('address2')
            ->setDefined('address3')
            ->setDefined('latitude')
            ->setDefined('longitude');
    }
}
