<?php

declare(strict_types=1);

namespace Dedi\SyliusShipmentPlugin\DependencyInjection\Compiler;

use Dedi\SyliusShipmentPlugin\Provider\CachedProvider;
use Dedi\SyliusShipmentPlugin\Provider\LocalProvider;
use InvalidArgumentException;
use Symfony\Component\DependencyInjection\Compiler\CompilerPassInterface;
use Symfony\Component\DependencyInjection\ContainerBuilder;
use Symfony\Component\DependencyInjection\Definition;
use Symfony\Component\DependencyInjection\Reference;

final class RegisterProvidersPass implements CompilerPassInterface
{
    public function process(ContainerBuilder $container): void
    {
        if (!$container->hasDefinition('dedi_sylius_shipment_plugin.registry.provider')) {
            return;
        }

        $registry = $container->getDefinition('dedi_sylius_shipment_plugin.registry.provider');
        $cacheEnabled = $container->getParameter('dedi_sylius_shipment_plugin.cache.enabled') === true;
        $localEnabled = $container->getParameter('dedi_sylius_shipment_plugin.local') === true;

        $typeToLabelMap = [];
        foreach ($container->findTaggedServiceIds('dedi_sylius_shipment_plugin.provider') as $id => $tagged) {
            foreach ($tagged as $attributes) {
                if (!isset($attributes['code'], $attributes['label'])) {
                    throw new InvalidArgumentException('Tagged pickup point provider `' . $id . '` needs to have `code`, and `label` attributes.');
                }

                $typeToLabelMap[$attributes['code']] = $attributes['label'];

                if ($cacheEnabled) {
                    $decoratedId = $id;
                    $id .= '.cached'; // overwrite the id
                    $cachedDefinition = new Definition(CachedProvider::class, [
                        new Reference('dedi_sylius_shipment_plugin.cache'),
                        new Reference($id . '.inner'),
                    ]);
                    $cachedDefinition->setDecoratedService($decoratedId, null, 256);

                    $container->setDefinition($id, $cachedDefinition);
                }

                if ($localEnabled) {
                    $decoratedId = $id;
                    $id .= '.local'; // overwrite the id
                    $cachedDefinition = new Definition(LocalProvider::class, [
                        new Reference($id . '.inner'),
                        new Reference('dedi_sylius_shipment_plugin.repository.pickup_point'),
                    ]);
                    $cachedDefinition->setDecoratedService($decoratedId, null, 512);

                    $container->setDefinition($id, $cachedDefinition);
                }

                $registry->addMethodCall('register', [$attributes['code'], new Reference($id)]);
            }
        }

        $container->setParameter('dedi_sylius_shipment_plugin.providers', $typeToLabelMap);
    }
}
