<?php

declare(strict_types=1);

namespace Dedi\SyliusReorderPlugin\Processor;

use Dedi\SyliusReorderPlugin\Factory\WrongItemsFactoryInterface;
use Dedi\SyliusReorderPlugin\Model\WrongItemsInterface;
use Sylius\Component\Core\Model\OrderInterface;
use Sylius\Component\Core\Model\OrderItemInterface;
use Sylius\Component\Core\Model\ProductInterface;
use Sylius\Component\Core\Model\ProductVariantInterface;
use Sylius\Component\Inventory\Checker\AvailabilityCheckerInterface;
use Sylius\Component\Order\Modifier\OrderItemQuantityModifierInterface;
use Sylius\Component\Order\Modifier\OrderModifierInterface;
use Sylius\Component\Resource\Factory\FactoryInterface;

class ReorderProcessor implements ReorderProcessorInterface
{
    public function __construct(
        private readonly OrderItemQuantityModifierInterface $orderItemQuantityModifier,
        private readonly OrderModifierInterface $orderModifier,
        private readonly AvailabilityCheckerInterface $availabilityChecker,
        private readonly FactoryInterface $orderItemFactory,
        private readonly WrongItemsFactoryInterface $wrongItemFactory,
    ) {
    }

    public function processItems(OrderInterface $order, OrderInterface $reorder): WrongItemsInterface
    {
        $wrongItems = $this->wrongItemFactory->createNew();

        $orderItems = $order->getItems();

        foreach ($orderItems as $orderItem) {
            $variant = $orderItem->getVariant();

            if (!$variant instanceof ProductVariantInterface ||
                !$this->availabilityChecker->isStockAvailable($variant)
            ) {
                $wrongItems->addUnavailableItem($orderItem);

                continue;
            }

            /** @var OrderItemInterface $newItem */
            $newItem = $this->orderItemFactory->createNew();
            $product = $variant->getProduct();

            if (!$product instanceof ProductInterface) {
                $wrongItems->addUnavailableItem($orderItem);

                continue;
            }

            $newItem->setVariant($variant);
            $newItem->setUnitPrice($orderItem->getUnitPrice());
            $newItem->setProductName($product->getName());
            $newItem->setVariantName($variant->getName());

            $this->orderItemQuantityModifier->modify($newItem, $orderItem->getQuantity());
            $this->orderModifier->addToOrder($reorder, $newItem);

            if ($orderItem->getUnitPrice() !== $newItem->getUnitPrice()) {
                $wrongItems->addPriceUpdatedItem($orderItem);
            }
        }

        return $wrongItems;
    }

    public function processOrderData(OrderInterface $order, OrderInterface $reorder): void
    {
        $billingAddress = $order->getBillingAddress();
        $shippingAddress = $order->getShippingAddress();

        if (null !== $billingAddress) {
            $reorder->setBillingAddress(clone $billingAddress);
        }

        if (null !== $shippingAddress) {
            $reorder->setShippingAddress(clone $shippingAddress);
        }
    }
}
