<?php

declare(strict_types=1);

namespace Dedi\SyliusReorderPlugin\Controller;

use Dedi\SyliusReorderPlugin\ErrorHandler\ReorderItemErrorHandlerInterface;
use Dedi\SyliusReorderPlugin\Manager\ReorderManagerInterface;
use Psr\Log\LoggerInterface;
use Sylius\Component\Core\Model\OrderInterface;
use Sylius\Component\Core\Repository\OrderRepositoryInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Session\Session;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Symfony\Contracts\Translation\TranslatorInterface;

class ReorderController extends AbstractController
{
    /** @param OrderRepositoryInterface<OrderInterface> $orderRepository */
    public function __construct(
        private readonly UrlGeneratorInterface $urlGenerator,
        private readonly OrderRepositoryInterface $orderRepository,
        private readonly TranslatorInterface $translator,
        private readonly ReorderManagerInterface $reorderManager,
        private readonly LoggerInterface $logger,
        private readonly ReorderItemErrorHandlerInterface $reorderItemErrorHandler,
    ) {
    }

    public function __invoke(Request $request): RedirectResponse
    {
        $tokenValue = $request->attributes->getString('tokenValue');
        $order = $this->orderRepository->findOneByTokenValue($tokenValue);

        /** @var Session $session */
        $session = $request->getSession();
        $flashBag = $session->getFlashBag();

        if (!$order instanceof OrderInterface) {
            $flashBag->add('error', $this->translator->trans('dedi_sylius_reorder.ui.order_not_found'));

            $this->logger->error(sprintf('Order with tokenValue %s not found on reorder processing', $tokenValue));

            return new RedirectResponse($this->urlGenerator->generate('sylius_shop_account_order_index'));
        }

        try {
            $wrongItems = $this->reorderManager->reorder($order);
        } catch (\Exception $exception) {
            $flashBag->add('error', $this->translator->trans('dedi_sylius_reorder.ui.error_occurred'));

            $this->logger->error($exception->getMessage(), ['exception' => $exception]);

            return new RedirectResponse($this->urlGenerator->generate('sylius_shop_account_order_index'));
        }

        if (!$wrongItems->getUnavailableItems()->isEmpty()) {
            $message = $this->translator->trans('dedi_sylius_reorder.ui.unavailable_items', domain: 'flashes');
            $this->reorderItemErrorHandler->handle($message, $wrongItems->getUnavailableItems()->toArray());
        }
        if (!$wrongItems->getPriceUpdatedItems()->isEmpty()) {
            $message = $this->translator->trans('dedi_sylius_reorder.ui.price_updated_items', domain: 'flashes');
            $this->reorderItemErrorHandler->handle($message, $wrongItems->getPriceUpdatedItems()->toArray(), 'info');
        }

        return new RedirectResponse($this->urlGenerator->generate('sylius_shop_cart_summary'));
    }
}
