<?php

declare(strict_types=1);

namespace Dedi\SyliusReorderPlugin\Api\Shop\Controller;

use Dedi\SyliusReorderPlugin\Api\Enum\ErrorMessage;
use Dedi\SyliusReorderPlugin\Api\ErrorHandler\ReorderItemErrorHandlerInterface;
use Dedi\SyliusReorderPlugin\Manager\ReorderManagerInterface;
use Sylius\Component\Core\Model\OrderInterface;
use Sylius\Component\Core\Repository\OrderRepositoryInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Security\Core\Authentication\Token\Storage\TokenStorageInterface;
use Symfony\Component\Security\Core\User\UserInterface;

class ReorderAction extends AbstractController
{
    /** @param OrderRepositoryInterface<OrderInterface> $orderRepository */
    public function __construct(
        private readonly OrderRepositoryInterface $orderRepository,
        private readonly ReorderManagerInterface $reorderManager,
        private readonly TokenStorageInterface $tokenStorage,
        private readonly ReorderItemErrorHandlerInterface $reorderItemErrorHandler,
    ) {
    }

    public function __invoke(string $tokenValue): JsonResponse
    {
        $token = $this->tokenStorage->getToken();
        $user = $token?->getUser();

        if (!$user instanceof UserInterface) {
            return new JsonResponse(status: Response::HTTP_UNAUTHORIZED);
        }

        $order = $this->orderRepository->findOneByTokenValue($tokenValue);

        if (!$order instanceof OrderInterface) {
            return new JsonResponse(ErrorMessage::ORDER_NOT_FOUND->value, Response::HTTP_NOT_FOUND);
        }

        if ($order->getUser() !== $user) {
            return new JsonResponse(status: Response::HTTP_FORBIDDEN);
        }

        try {
            $wrongItems = $this->reorderManager->reorder($order);
        } catch (\Exception $exception) {
            return new JsonResponse($exception->getMessage(), Response::HTTP_INTERNAL_SERVER_ERROR);
        }

        $unavailableItems = $this->reorderItemErrorHandler->handle($wrongItems->getUnavailableItems()->toArray());
        $priceUpdatedItems = $this->reorderItemErrorHandler->handle($wrongItems->getPriceUpdatedItems()->toArray());

        $response = [];

        if (count($unavailableItems) > 0) {
            $response[ErrorMessage::UNAVAILABLE_ITEMS->value] = $unavailableItems;
        }
        if (count($priceUpdatedItems) > 0) {
            $response[ErrorMessage::PRICE_UPDATED_ITEMS->value] = $priceUpdatedItems;
        }

        if (count($response) > 0) {
            return new JsonResponse($response, Response::HTTP_OK);
        }

        return new JsonResponse(status: Response::HTTP_NO_CONTENT);
    }
}
