<?php

declare(strict_types=1);

namespace Dedi\SyliusQuotePlugin\Context;

use Dedi\SyliusQuotePlugin\Entity\OrderInterface as DediQuoteOrderInterface;
use Dedi\SyliusQuotePlugin\Exception\QuoteNotFoundException;
use Dedi\SyliusQuotePlugin\Factory\QuoteFactoryInterface;
use InvalidArgumentException;
use Sylius\Component\Channel\Context\ChannelNotFoundException;
use Sylius\Component\Core\Context\ShopperContextInterface;
use Sylius\Component\Core\Model\ChannelInterface;
use Sylius\Component\Core\Model\CustomerInterface;
use Sylius\Component\Currency\Context\CurrencyNotFoundException;
use Sylius\Component\Locale\Context\LocaleNotFoundException;
use Sylius\Component\Order\Model\OrderInterface;

class ShopBasedQuoteCartContext implements QuoteCartContextInterface
{
    private readonly QuoteFactoryInterface $quoteFactory;

    private readonly ShopperContextInterface $shopperContext;

    private ?OrderInterface $quote;

    public function __construct(QuoteFactoryInterface $quoteFactory, ShopperContextInterface $shopperContext)
    {
        $this->quoteFactory = $quoteFactory;
        $this->shopperContext = $shopperContext;
        $this->quote = null;
    }

    public function getCart(): OrderInterface
    {
        if (null !== $this->quote) {
            return $this->quote;
        }

        $quote = $this->quoteFactory->createNew();
        if (!$quote instanceof DediQuoteOrderInterface) {
            throw new InvalidArgumentException(sprintf(
                'Expected an instance of "%s", got "%s"',
                DediQuoteOrderInterface::class,
                get_class($quote),
            ));
        }

        try {
            /** @var ChannelInterface $channel */
            $channel = $this->shopperContext->getChannel();

            $quote->setChannel($channel);
            $quote->setCurrencyCode($channel->getBaseCurrency()->getCode());
            $quote->setLocaleCode($this->shopperContext->getLocaleCode());
        } catch (ChannelNotFoundException | CurrencyNotFoundException | LocaleNotFoundException $exception) {
            throw new QuoteNotFoundException('Sylius was not able to prepare the quote.', $exception);
        }

        /** @var CustomerInterface|null $customer */
        $customer = $this->shopperContext->getCustomer();
        if (null !== $customer) {
            $quote->setCustomer($customer);
        }

        $this->quote = $quote;

        return $quote;
    }

    public function reset(): void
    {
        $this->quote = null;
    }
}
