<?php

declare(strict_types=1);

namespace Dedi\SyliusQuotePlugin\Storage;

use Dedi\SyliusQuotePlugin\Entity\OrderInterface;
use Dedi\SyliusQuotePlugin\Repository\QuoteRepositoryInterface;
use Sylius\Component\Core\Model\ChannelInterface;
use Symfony\Component\HttpFoundation\RequestStack;

class QuoteCartSessionStorage implements QuoteCartStorageInterface
{
    private readonly RequestStack $requestStack;

    private readonly string $sessionKeyName;

    private readonly QuoteRepositoryInterface $quoteRepository;

    public function __construct(
        RequestStack $requestStack,
        string $sessionKeyName,
        QuoteRepositoryInterface $quoteRepository,
    ) {
        $this->requestStack = $requestStack;
        $this->sessionKeyName = $sessionKeyName;
        $this->quoteRepository = $quoteRepository;
    }

    public function hasForChannel(ChannelInterface $channel): bool
    {
        $session = $this->requestStack->getSession();

        return $session->has($this->getQuoteKeyName($channel));
    }

    public function getForChannel(ChannelInterface $channel): ?OrderInterface
    {
        if ($this->hasForChannel($channel)) {
            $session = $this->requestStack->getSession();
            $quoteId = $session->get($this->getQuoteKeyName($channel));
            /** @var ?OrderInterface $quote */
            $quote = $this->quoteRepository->findCartByChannel($quoteId, $channel);

            return $quote;
        }

        return null;
    }

    public function setForChannel(ChannelInterface $channel, OrderInterface $quote): void
    {
        $session = $this->requestStack->getSession();
        $session->set($this->getQuoteKeyName($channel), $quote->getId());
    }

    public function removeForChannel(ChannelInterface $channel): void
    {
        $session = $this->requestStack->getSession();
        $session->remove($this->getQuoteKeyName($channel));
    }

    private function getQuoteKeyName(ChannelInterface $channel): string
    {
        return sprintf('%s.%s', $this->sessionKeyName, $channel->getCode());
    }
}
