<?php

declare(strict_types=1);

namespace Dedi\SyliusQuotePlugin\Menu;

use Dedi\SyliusQuotePlugin\Entity\OrderInterface;
use Dedi\SyliusQuotePlugin\Enum\QuoteTransitionEnum;
use InvalidArgumentException;
use Knp\Menu\FactoryInterface;
use Knp\Menu\ItemInterface;
use SM\Factory\FactoryInterface as StateMachineFactoryInterface;
use Symfony\Component\Security\Csrf\CsrfTokenManagerInterface;

class QuoteShowMenuBuilder
{
    private readonly FactoryInterface $menuItemFactory;

    private readonly StateMachineFactoryInterface $stateMachineFactory;

    private readonly CsrfTokenManagerInterface $csrfTokenManager;

    public function __construct(
        FactoryInterface $menuItemFactory,
        StateMachineFactoryInterface $stateMachineFactory,
        CsrfTokenManagerInterface $csrfTokenManager,
    ) {
        $this->menuItemFactory = $menuItemFactory;
        $this->stateMachineFactory = $stateMachineFactory;
        $this->csrfTokenManager = $csrfTokenManager;
    }

    public function createMenu(array $options): ItemInterface
    {
        $menu = $this->menuItemFactory->createItem('root');

        if (!isset($options['quote'])) {
            return $menu;
        }

        /** @var OrderInterface|mixed $quote */
        $quote = $options['quote'];
        if (!$quote instanceof OrderInterface) {
            throw new InvalidArgumentException(sprintf(
                'Expected an instance of "%s", got "%s"',
                OrderInterface::class,
                get_class($quote),
            ));
        }

        $menu
            ->addChild('dedi_sylius_quote_quote_update', [
                'route' => 'dedi_sylius_quote_admin_quote_information_update',
                'routeParameters' => ['id' => $quote->getQuoteInformation()->getId()],
            ])
            ->setAttribute('type', 'link')
            ->setLabel('sylius.ui.update')
            ->setLabelAttribute('icon', 'pencil')
        ;

        $stateMachine = $this->stateMachineFactory->get($quote, QuoteTransitionEnum::GRAPH);
        $csrfToken = $this->csrfTokenManager->getToken((string) $quote->getId())->getValue();

        if ($stateMachine->can(QuoteTransitionEnum::TRANSITIONS_ACCEPT)) {
            $menu
                ->addChild('dedi_sylius_quote_accept', [
                    'route' => 'dedi_sylius_quote_admin_quote_accept',
                    'routeParameters' => ['id' => $quote->getId(), '_csrf_token' => $csrfToken],
                ])
                ->setAttribute('type', 'transition')
                ->setAttribute('confirmation', true)
                ->setLabel('dedi_sylius_quote.ui.accept')
                ->setLabelAttribute('color', 'olive')
                ->setLabelAttribute('icon', 'check')
            ;
        }

        if ($stateMachine->can(QuoteTransitionEnum::TRANSITIONS_ACCEPT_RESUPPLY)) {
            $menu
                ->addChild('dedi_sylius_quote_accept_resupply', [
                    'route' => 'dedi_sylius_quote_admin_quote_accept_resupply',
                    'routeParameters' => ['id' => $quote->getId(), '_csrf_token' => $csrfToken],
                ])
                ->setAttribute('type', 'transition')
                ->setAttribute('confirmation', true)
                ->setLabel('dedi_sylius_quote.ui.accept_resupply')
                ->setLabelAttribute('color', 'yellow')
                ->setLabelAttribute('icon', 'check')
            ;
        }

        if ($stateMachine->can(QuoteTransitionEnum::TRANSITIONS_DENY)) {
            $menu
                ->addChild('dedi_sylius_quote_deny', [
                    'route' => 'dedi_sylius_quote_admin_quote_deny',
                    'routeParameters' => ['id' => $quote->getId(), '_csrf_token' => $csrfToken],
                ])
                ->setAttribute('type', 'transition')
                ->setAttribute('confirmation', true)
                ->setLabel('dedi_sylius_quote.ui.deny')
                ->setLabelAttribute('color', 'red')
                ->setLabelAttribute('icon', 'ban')
            ;
        }

        return $menu;
    }
}
