<?php

declare(strict_types=1);

namespace Dedi\SyliusQuotePlugin\Factory;

use Dedi\SyliusQuotePlugin\Entity\OrderItemInterface;
use Dedi\SyliusQuotePlugin\Entity\QuoteItemInformationInterface;
use Sylius\Component\Channel\Context\ChannelContextInterface;
use Sylius\Component\Core\Calculator\ProductVariantPricesCalculatorInterface;
use Sylius\Component\Product\Model\ProductInterface;
use Sylius\Component\Product\Resolver\ProductVariantResolverInterface;
use Sylius\Component\Resource\Factory\FactoryInterface;

class QuoteItemFactory implements QuoteItemFactoryInterface
{
    private readonly FactoryInterface $orderItemFactory;

    private readonly ProductVariantResolverInterface $productVariantResolver;

    private readonly FactoryInterface $quoteItemInformationFactory;

    private readonly ProductVariantPricesCalculatorInterface $productVariantPricesCalculator;

    private readonly ChannelContextInterface $channelContext;

    public function __construct(
        FactoryInterface $orderItemFactory,
        ProductVariantResolverInterface $productVariantResolver,
        FactoryInterface $quoteItemInformationFactory,
        ProductVariantPricesCalculatorInterface $productVariantPricesCalculator,
        ChannelContextInterface $channelContext,
    ) {
        $this->orderItemFactory = $orderItemFactory;
        $this->productVariantResolver = $productVariantResolver;
        $this->quoteItemInformationFactory = $quoteItemInformationFactory;
        $this->productVariantPricesCalculator = $productVariantPricesCalculator;
        $this->channelContext = $channelContext;
    }

    public function createNew(): OrderItemInterface
    {
        $quoteItem = $this->orderItemFactory->createNew();

        if (!$quoteItem instanceof OrderItemInterface) {
            throw new \InvalidArgumentException('The order factory must create an instance of OrderInterface');
        }

        return $quoteItem;
    }

    public function createForProduct(ProductInterface $product): OrderItemInterface
    {
        $quoteItem = $this->createNew();
        $variant = $this->productVariantResolver->getVariant($product);

        if (null !== $variant) {
            $quoteItem->setVariant($variant);
            $quoteItem->setVariantName($variant->getName());

            $channel = $this->channelContext->getChannel();
            $unitPrice = $this->productVariantPricesCalculator->calculate(
                $variant,
                ['channel' => $channel],
            );

            /** @var QuoteItemInformationInterface $quoteItemInformation */
            $quoteItemInformation = $this->quoteItemInformationFactory->createNew();
            $quoteItemInformation->setUnitPrice($unitPrice);

            $quoteItem->setQuoteItemInformation($quoteItemInformation);
        }

        $quoteItem->setProductName($product->getName());

        return $quoteItem;
    }
}
