<?php

declare(strict_types=1);

namespace Dedi\SyliusQuotePlugin\Command;

use DateTime;
use Dedi\SyliusQuotePlugin\Service\ExpiredQuoteRemoverInterface;
use InvalidArgumentException;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Command\LockableTrait;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;

class RemoveExpiredQuoteCommand extends Command
{
    use LockableTrait;

    private readonly ExpiredQuoteRemoverInterface $expiredQuoteDeleter;

    protected const DATE_PATTERN = '/^[\d]{2}\-[\d]{2}\-[\d]{4}$/';

    public function __construct(string $name = null, ExpiredQuoteRemoverInterface $expiredQuoteDeleter)
    {
        parent::__construct($name);

        $this->expiredQuoteDeleter = $expiredQuoteDeleter;
    }

    protected function configure(): void
    {
        $this
            ->setDescription('Applies the "expired" transition on quote if their expiration date matches the given date')
            ->addArgument('date', InputArgument::OPTIONAL, 'The expiration date (format : dd-mm-yyyy). Defaults to the current date.')
        ;
    }

    public function run(InputInterface $input, OutputInterface $output): int
    {
        if (!$this->lock()) {
            $output->writeln('The command is already running in another process.');

            return Command::SUCCESS;
        }

        $date = new DateTime();
        if ($input->hasArgument('date') && '' !== $dateParameter = $input->getArgument('date')) {
            if (!preg_match(self::DATE_PATTERN, (string) $dateParameter)) {
                throw new InvalidArgumentException(sprintf(
                    'The value %s does not match the expected pattern.',
                    $dateParameter,
                ));
            }

            $date = DateTime::createFromFormat('d-m-Y', $dateParameter);
        }

        $this->expiredQuoteDeleter->deleteExpiredQuotes($date);

        $this->release();

        return Command::SUCCESS;
    }
}
