<?php

declare(strict_types=1);

namespace Dedi\SyliusQuotePlugin\Form\Type;

use Dedi\SyliusQuotePlugin\Entity\OrderInterface;
use Dedi\SyliusQuotePlugin\Entity\OrderItemInterface;
use Sylius\Bundle\ResourceBundle\Form\Type\AbstractResourceType;
use Sylius\Bundle\ResourceBundle\Form\Type\ResourceAutocompleteChoiceType;
use Sylius\Component\Core\Calculator\ProductVariantPricesCalculatorInterface;
use Sylius\Component\Order\Modifier\OrderItemQuantityModifierInterface;
use Sylius\Component\Order\Processor\OrderProcessorInterface;
use Symfony\Component\Form\Extension\Core\Type\HiddenType;
use Symfony\Component\Form\Extension\Core\Type\IntegerType;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\Form\FormEvent;
use Symfony\Component\Form\FormEvents;
use Symfony\Component\OptionsResolver\OptionsResolver;
use Symfony\Component\Validator\Constraints\NotBlank;
use Symfony\Component\Validator\Constraints\Range;

class OrderItemType extends AbstractResourceType
{
    private OrderItemQuantityModifierInterface $orderItemQuantityModifier;

    private OrderProcessorInterface $orderProcessor;

    private ProductVariantPricesCalculatorInterface $productVariantPricesCalculator;

    public function __construct(
        OrderItemQuantityModifierInterface $orderItemQuantityModifier,
        OrderProcessorInterface $orderProcessor,
        ProductVariantPricesCalculatorInterface $productVariantPricesCalculator,
        string $dataClass,
        array $validationGroups = [],
    ) {
        parent::__construct($dataClass, $validationGroups);
        $this->productVariantPricesCalculator = $productVariantPricesCalculator;
        $this->orderItemQuantityModifier = $orderItemQuantityModifier;
        $this->orderProcessor = $orderProcessor;
    }

    public function buildForm(FormBuilderInterface $builder, array $options): void
    {
        $builder
            ->add('variant', ResourceAutocompleteChoiceType::class, [
                'label' => false,
                'resource' => 'sylius.product_variant',
                'choice_name' => 'descriptor',
                'choice_value' => 'id',
                'constraints' => [
                    new NotBlank(['groups' => $this->validationGroups]),
                ],
            ])
            ->add('originalQuantity', HiddenType::class, [
                'mapped' => false,
            ])
            ->add('quoteItemInformation', QuoteItemInformationType::class, [
                'mapped' => true,
            ])
            ->add('quantity', IntegerType::class, [
                'attr' => ['min' => 1],
                'label' => false,
                'empty_data' => 1,
                'constraints' => [
                    new Range(['min' => 1, 'groups' => $this->validationGroups]),
                ],
            ])
        ;

        $builder->addEventListener(FormEvents::POST_SUBMIT, $this->setProductAndVariantName(...));
        $builder->addEventListener(FormEvents::POST_SET_DATA, function (FormEvent $event) {
            $orderItem = $event->getData();

            if (null === $orderItem) {
                return;
            }

            $form = $event->getForm();
            $form->get('originalQuantity')->setData($orderItem->getQuantity());
        });
    }

    public function setProductAndVariantName(FormEvent $event): void
    {
        /** @var OrderItemInterface $quoteItem */
        $quoteItem = $event->getData();

        $oldQuantity = $event->getForm()->get('originalQuantity')->getData() ?? 0;
        $oldQuantity = (int) $oldQuantity;
        $newQuantity = $quoteItem->getQuantity();

        if (null === ($variant = $quoteItem->getVariant())) {
            return;
        }

        $quoteItem->setProductName($variant->getProduct()->getName());
        $quoteItem->setVariantName($variant->getName());

        if (null === ($order = $quoteItem->getOrder() ?? $event->getForm()->getParent()->getParent()->getData())) {
            return;
        }

        if (null === $variant->getChannelPricingForChannel($order->getChannel())) {
            return;
        }

        $quoteItem->setOriginalUnitPrice($this->productVariantPricesCalculator->calculateOriginal($variant, ['channel' => $order->getChannel()]));

        $quoteItem->setQuantity($oldQuantity);
        $this->orderItemQuantityModifier->modify($quoteItem, $newQuantity);
        $quoteItem->setQuantity($newQuantity);

        $quoteItem->setImmutable(true);

        $this->orderProcessor->process($order);
    }

    public function configureOptions(OptionsResolver $resolver): void
    {
        parent::configureOptions($resolver);

        $resolver
            ->setDefined(['currency'])
            ->setRequired('quote')
            ->setAllowedTypes('quote', [OrderInterface::class])
        ;
    }

    public function getBlockPrefix(): string
    {
        return 'dedi_sylius_quote_item';
    }
}
