<?php

declare(strict_types=1);

namespace Dedi\SyliusQuotePlugin\Entity;

use Dedi\SyliusQuotePlugin\Enum\QuoteInformationStateEnum;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;

trait QuoteOrderTrait
{
    abstract public function getState(): string;

    /** @ORM\Column(name="quote_state", type="string") */
    protected string $quoteState = QuoteInformationStateEnum::STATE_NONE;

    /** @ORM\OneToOne(targetEntity="Dedi\SyliusQuotePlugin\Entity\QuoteInformationInterface", mappedBy="order", cascade={"persist", "remove"}) */
    protected ?QuoteInformationInterface $quoteInformation = null;

    /**
     * @ORM\ManyToOne(targetEntity="Sylius\Component\Order\Model\OrderInterface", inversedBy="orders")
     *
     * @ORM\JoinColumn(name="quote_id", referencedColumnName="id", nullable=true, onDelete="SET NULL")
     */
    private ?OrderInterface $quote = null;

    /** @ORM\OneToMany(targetEntity="Sylius\Component\Order\Model\OrderInterface", mappedBy="quote") */
    private Collection $orders;

    public function initializeQuoteOrders(): void
    {
        $this->orders = new ArrayCollection();
    }

    public function getQuoteInformation(): ?QuoteInformationInterface
    {
        return $this->quoteInformation;
    }

    public function setQuoteInformation(?QuoteInformationInterface $quoteInformation = null): self
    {
        $this->quoteInformation = $quoteInformation;
        if (null !== $quoteInformation) {
            $quoteInformation->setOrder($this);
        }

        return $this;
    }

    public function getQuote(): ?OrderInterface
    {
        return $this->quote;
    }

    public function setQuote(?OrderInterface $quote): self
    {
        $this->quote = $quote;

        return $this;
    }

    public function getOrders(): Collection
    {
        return $this->orders;
    }

    public function hasOrder(OrderInterface $order): bool
    {
        return $this->orders->contains($order);
    }

    public function addOrder(OrderInterface $order): self
    {
        if (!$this->hasOrder($order)) {
            $this->orders[] = $order;
            $order->setQuote($this);
        }

        return $this;
    }

    public function removeOrder(OrderInterface $order): self
    {
        if ($this->hasOrder($order)) {
            $this->orders->removeElement($order);
            if ($order->getQuote() === $this) {
                $order->setQuote(null);
            }
        }

        return $this;
    }

    public function getQuoteState(): string
    {
        return $this->quoteState;
    }

    public function setQuoteState(string $quoteState): self
    {
        $this->quoteState = $quoteState;

        return $this;
    }
}
