<?php

declare(strict_types=1);

namespace Dedi\SyliusQuotePlugin\Controller;

use Dedi\SyliusQuotePlugin\Converter\QuoteToCartConverterInterface;
use Dedi\SyliusQuotePlugin\Entity\OrderInterface;
use Dedi\SyliusQuotePlugin\Enum\QuoteActionEnum;
use Dedi\SyliusQuotePlugin\Generator\QuotePdfFileGeneratorInterface;
use Dedi\SyliusQuotePlugin\Specification\Order\IsQuoteSpecificationInterface;
use Dedi\SyliusQuotePlugin\Voter\QuoteVoter;
use Doctrine\Persistence\ObjectManager;
use Sylius\Bundle\CoreBundle\Controller\OrderController;
use Sylius\Bundle\ResourceBundle\Controller\AuthorizationCheckerInterface;
use Sylius\Bundle\ResourceBundle\Controller\EventDispatcherInterface;
use Sylius\Bundle\ResourceBundle\Controller\FlashHelperInterface;
use Sylius\Bundle\ResourceBundle\Controller\NewResourceFactoryInterface;
use Sylius\Bundle\ResourceBundle\Controller\RedirectHandlerInterface;
use Sylius\Bundle\ResourceBundle\Controller\RequestConfigurationFactoryInterface;
use Sylius\Bundle\ResourceBundle\Controller\ResourceDeleteHandlerInterface;
use Sylius\Bundle\ResourceBundle\Controller\ResourceFormFactoryInterface;
use Sylius\Bundle\ResourceBundle\Controller\ResourcesCollectionProviderInterface;
use Sylius\Bundle\ResourceBundle\Controller\ResourceUpdateHandlerInterface;
use Sylius\Bundle\ResourceBundle\Controller\SingleResourceProviderInterface;
use Sylius\Bundle\ResourceBundle\Controller\StateMachineInterface;
use Sylius\Bundle\ResourceBundle\Controller\ViewHandlerInterface;
use Sylius\Component\Channel\Context\ChannelContextInterface;
use Sylius\Component\Core\Model\ChannelInterface;
use Sylius\Component\Core\Storage\CartStorageInterface;
use Sylius\Component\Order\Context\CartContextInterface;
use Sylius\Component\Order\Repository\OrderRepositoryInterface;
use Sylius\Component\Resource\Factory\FactoryInterface;
use Sylius\Component\Resource\Metadata\Metadata;
use Sylius\Component\Resource\Metadata\MetadataInterface;
use Sylius\Component\Resource\Repository\RepositoryInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Webmozart\Assert\Assert;

class QuoteController extends OrderController
{
    public function __construct(MetadataInterface $metadata, RequestConfigurationFactoryInterface $requestConfigurationFactory, ?ViewHandlerInterface $viewHandler, RepositoryInterface $repository, FactoryInterface $factory, NewResourceFactoryInterface $newResourceFactory, ObjectManager $manager, SingleResourceProviderInterface $singleResourceProvider, ResourcesCollectionProviderInterface $resourcesFinder, ResourceFormFactoryInterface $resourceFormFactory, RedirectHandlerInterface $redirectHandler, FlashHelperInterface $flashHelper, AuthorizationCheckerInterface $authorizationChecker, EventDispatcherInterface $eventDispatcher, ?StateMachineInterface $stateMachine, ResourceUpdateHandlerInterface $resourceUpdateHandler, ResourceDeleteHandlerInterface $resourceDeleteHandler)
    {
        // Overrides the metadata to use the correct alias
        $metadata = Metadata::fromAliasAndConfiguration('dedi_sylius_quote.quote', $metadata->getParameters());
        parent::__construct($metadata, $requestConfigurationFactory, $viewHandler, $repository, $factory, $newResourceFactory, $manager, $singleResourceProvider, $resourcesFinder, $resourceFormFactory, $redirectHandler, $flashHelper, $authorizationChecker, $eventDispatcher, $stateMachine, $resourceUpdateHandler, $resourceDeleteHandler);
    }

    public function getContext(): CartContextInterface
    {
        /** @var CartContextInterface $cartContext */
        $cartContext = $this->get('dedi_sylius_quote.context.quote');

        return $cartContext;
    }

    protected function getOrderRepository(): OrderRepositoryInterface
    {
        /** @var OrderRepositoryInterface $repository */
        $repository = $this->get('dedi_sylius_quote.repository.quote');

        return $repository;
    }

    public function thankYouAction(Request $request): Response
    {
        $configuration = $this->requestConfigurationFactory->create($this->metadata, $request);

        $quoteId = $request->getSession()->get('dedi_sylius_quote_quote_id', null);

        if (null === $quoteId) {
            $options = $configuration->getParameters()->get('after_failure');

            return $this->redirectHandler->redirectToRoute(
                $configuration,
                $options['route'] ?? 'sylius_shop_homepage',
                $options['parameters'] ?? [],
            );
        }

        $request->getSession()->remove('dedi_sylius_quote_quote_id');
        $quote = $this->repository->find($quoteId);
        Assert::notNull($quote);

        return $this->render(
            $configuration->getParameters()->get('template'),
            [
                'quote' => $quote,
            ],
        );
    }

    public function downloadPdfAction(
        Request $request,
        string $number,
        QuotePdfFileGeneratorInterface $quotePdfFileGenerator,
    ): Response {
        $configuration = $this->requestConfigurationFactory->create($this->metadata, $request);

        $this->isGrantedOr403($configuration, QuoteActionEnum::DOWNLOAD);

        $repository = $this->getOrderRepository();

        /** @var OrderInterface|null $quote */
        $quote = $repository->findOneByNumber($number);
        if (null === $quote) {
            throw $this->createNotFoundException();
        }

        $this->denyAccessUnlessGranted(QuoteVoter::IS_QUOTE_USABLE, $quote);

        $quotePdf = $quotePdfFileGenerator->generate($quote);

        $response = new Response($quotePdf->getContent(), Response::HTTP_OK, ['Content-Type' => 'application/pdf']);
        $response->headers->add([
            'Content-Disposition' => $response->headers->makeDisposition('inline', $quotePdf->getFileName()),
        ]);

        return $response;
    }

    public function convertToCartAction(
        Request $request,
        CartContextInterface $cartContext,
        OrderRepositoryInterface $orderRepository,
        CartStorageInterface $cartStorage,
        ChannelContextInterface $channelContext,
        IsQuoteSpecificationInterface $isQuoteSpecification,
        QuoteToCartConverterInterface $quoteToCartConverter,
    ): Response {
        $configuration = $this->requestConfigurationFactory->create($this->metadata, $request);

        /** @var OrderInterface $quote */
        $quote = $this->findOr404($configuration);

        $this->denyAccessUnlessGranted(QuoteVoter::IS_QUOTE_ACCEPTED, $quote);

        /** @var OrderInterface $cart */
        $cart = $cartContext->getCart();
        if (!$cart->isEmpty() && !$isQuoteSpecification->isSatisfiedBy($cart)) {
            $orderRepository->remove($cart);
            $cart = $cartContext->getCart();
        }

        $quoteToCartConverter->convert($quote, $cart);
        $orderRepository->add($cart);

        /** @var ChannelInterface $channel */
        $channel = $channelContext->getChannel();
        $cartStorage->setForChannel($channel, $quote);

        $this->flashHelper->addSuccessFlash($configuration, QuoteActionEnum::CONVERT, $quote);

        return $this->redirectHandler->redirectToResource($configuration, $quote);
    }
}
