<?php

declare(strict_types=1);

namespace Dedi\SyliusQuotePlugin\Twig\FormComponent;

use Dedi\SyliusQuotePlugin\Enum\QuoteCartActionsEnum;
use Doctrine\Persistence\ObjectManager;
use Sylius\Bundle\UiBundle\Twig\Component\ResourceFormComponentTrait;
use Sylius\Bundle\UiBundle\Twig\Component\TemplatePropTrait;
use Sylius\Component\Core\Model\OrderInterface;
use Sylius\Component\Core\Repository\OrderRepositoryInterface;
use Sylius\Component\Order\SyliusCartEvents;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\EventDispatcher\GenericEvent;
use Symfony\Component\Form\FormFactoryInterface;
use Symfony\UX\LiveComponent\Attribute\AsLiveComponent;
use Symfony\UX\LiveComponent\Attribute\LiveAction;
use Symfony\UX\LiveComponent\Attribute\LiveArg;
use Symfony\UX\LiveComponent\Attribute\PreReRender;
use Symfony\UX\LiveComponent\ComponentToolsTrait;

#[AsLiveComponent]
class QuoteFormComponent
{
    use ComponentToolsTrait;

    /** @use ResourceFormComponentTrait<OrderInterface> */
    use ResourceFormComponentTrait;

    use TemplatePropTrait;

    public const SYLIUS_SHOP_QUOTE_CHANGED = 'sylius:shop:quote_changed';

    public const SYLIUS_SHOP_QUOTE_CLEARED = 'sylius:shop:quote_cleared';

    public bool $shouldSaveQuote = true;

    /** @param OrderRepositoryInterface<OrderInterface> $orderRepository */
    public function __construct(
        OrderRepositoryInterface $orderRepository,
        FormFactoryInterface $formFactory,
        string $resourceClass,
        string $formClass,
        protected readonly ObjectManager $manager,
        protected readonly EventDispatcherInterface $eventDispatcher,
    ) {
        $this->initialize($orderRepository, $formFactory, $resourceClass, $formClass);
    }

    #[PreReRender(priority: -100)]
    public function saveQuote(): void
    {
        if ($this->shouldSaveQuote) {
            $form = $this->getForm();
            if ($form->isValid()) {
                $this->eventDispatcher->dispatch(new GenericEvent($form->getData()), QuoteCartActionsEnum::CHANGE);
                $this->manager->flush();
                $this->emit(self::SYLIUS_SHOP_QUOTE_CHANGED, ['cartId' => $this->resource->getId()]);
            }
        }
    }

    #[LiveAction]
    public function removeItem(#[LiveArg] int $index): void
    {
        $data = $this->formValues['items'];
        unset($data[$index]);
        $this->formValues['items'] = array_values($data);

        $orderItem = $this->resource->getItems()->get($index);
        $this->eventDispatcher->dispatch(new GenericEvent($orderItem), SyliusCartEvents::CART_ITEM_REMOVE);

        $this->manager->persist($this->resource);
        $this->manager->flush();
        $this->manager->refresh($this->resource);

        $this->shouldSaveQuote = false;
        $this->submitForm();
        $this->emit(self::SYLIUS_SHOP_QUOTE_CHANGED, ['cartId' => $this->resource->getId()]);
    }

    #[LiveAction]
    public function clearQuote(): void
    {
        $this->formValues['items'] = [];
        $this->eventDispatcher->dispatch(new GenericEvent($this->resource), QuoteCartActionsEnum::CLEAR);
        $this->manager->remove($this->resource);
        $this->manager->flush();

        $this->shouldSaveQuote = false;
        $this->submitForm();
        $this->emit(self::SYLIUS_SHOP_QUOTE_CLEARED);
    }
}
