<?php

declare(strict_types=1);

namespace Dedi\SyliusQuotePlugin\Twig\FormComponent;

use Dedi\SyliusQuotePlugin\Context\QuoteCartContextInterface;
use Dedi\SyliusQuotePlugin\Entity\OrderItemInterface;
use Dedi\SyliusQuotePlugin\Factory\QuoteItemFactoryInterface;
use Doctrine\Persistence\ObjectManager;
use Sylius\Bundle\CoreBundle\Provider\FlashBagProvider;
use Sylius\Bundle\OrderBundle\Factory\AddToCartCommandFactoryInterface;
use Sylius\Bundle\ShopBundle\Twig\Component\Product\AddToCartFormComponent as BaseAddToCartFormComponent;
use Sylius\Bundle\ShopBundle\Twig\Component\Product\Trait\ProductLivePropTrait;
use Sylius\Bundle\ShopBundle\Twig\Component\Product\Trait\ProductVariantLivePropTrait;
use Sylius\Bundle\UiBundle\Twig\Component\TemplatePropTrait;
use Sylius\Component\Core\Factory\CartItemFactoryInterface;
use Sylius\Component\Core\Model\ProductInterface;
use Sylius\Component\Core\Model\ProductVariantInterface;
use Sylius\Component\Core\Repository\ProductRepositoryInterface;
use Sylius\Component\Core\Repository\ProductVariantRepositoryInterface;
use Sylius\Component\Order\Context\CartContextInterface;
use Sylius\Component\Order\SyliusCartEvents;
use Sylius\TwigHooks\LiveComponent\HookableLiveComponentTrait;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\EventDispatcher\GenericEvent;
use Symfony\Component\Form\FormFactoryInterface;
use Symfony\Component\Form\FormInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\Routing\RouterInterface;
use Symfony\UX\LiveComponent\Attribute\AsLiveComponent;
use Symfony\UX\LiveComponent\Attribute\LiveAction;
use Symfony\UX\LiveComponent\Attribute\LiveArg;
use Symfony\UX\LiveComponent\ComponentToolsTrait;
use Symfony\UX\LiveComponent\ComponentWithFormTrait;
use Symfony\UX\LiveComponent\DefaultActionTrait;

#[AsLiveComponent]
class AddToCartFormComponent extends BaseAddToCartFormComponent
{
    use ComponentToolsTrait;
    use ComponentWithFormTrait;
    use DefaultActionTrait;
    use HookableLiveComponentTrait;
    use ProductLivePropTrait;
    use ProductVariantLivePropTrait;
    use TemplatePropTrait;

    /**
     * @param CartItemFactoryInterface<OrderItemInterface> $cartItemFactory
     * @param QuoteItemFactoryInterface<OrderItemInterface> $quoteItemFactory
     * @param class-string $formClass
     * @param ProductRepositoryInterface<ProductInterface> $productRepository
     * @param ProductVariantRepositoryInterface<ProductVariantInterface> $productVariantRepository
     */
    public function __construct(
        FormFactoryInterface $formFactory,
        ObjectManager $manager,
        RouterInterface $router,
        RequestStack $requestStack,
        EventDispatcherInterface $eventDispatcher,
        CartContextInterface $cartContext,
        AddToCartCommandFactoryInterface $addToCartCommandFactory,
        CartItemFactoryInterface $cartItemFactory,
        string $formClass,
        ProductRepositoryInterface $productRepository,
        ProductVariantRepositoryInterface $productVariantRepository,
        protected readonly QuoteCartContextInterface $quoteCartContext,
        protected readonly QuoteItemFactoryInterface $quoteItemFactory,
    ) {
        parent::__construct(
            $formFactory,
            $manager,
            $router,
            $requestStack,
            $eventDispatcher,
            $cartContext,
            $addToCartCommandFactory,
            $cartItemFactory,
            $formClass,
            $productRepository,
            $productVariantRepository,
        );
    }

    #[LiveAction]
    public function addToQuote(
        #[LiveArg]
        bool $addFlashMessage = true,
    ): RedirectResponse {
        $this->form = $this->instantiateQuoteForm();
        $this->submitForm();
        $addToCartCommand = $this->getForm()->getData();

        $this->eventDispatcher->dispatch(new GenericEvent($addToCartCommand), SyliusCartEvents::CART_ITEM_ADD);
        $this->manager->persist($addToCartCommand->getCart());
        $this->manager->flush();

        if ($addFlashMessage) {
            FlashBagProvider::getFlashBag($this->requestStack)->add('success', 'dedi_sylius_quote.quote_item.add');
        }

        return new RedirectResponse($this->router->generate(
            'dedi_sylius_quote_shop_quote_summary',
        ));
    }

    protected function instantiateQuoteForm(): FormInterface
    {
        $addToCartCommand = $this->addToCartCommandFactory->createWithCartAndCartItem(
            $this->quoteCartContext->getCart(),
            $this->quoteItemFactory->createForProduct($this->product),
        );

        return $this->formFactory->create($this->formClass, $addToCartCommand, ['product' => $this->product]);
    }
}
