<?php

declare(strict_types=1);

namespace Dedi\SyliusQuotePlugin\Repository;

use Dedi\SyliusQuotePlugin\Enum\QuoteInformationStateEnum;
use Sylius\Bundle\CoreBundle\Doctrine\ORM\OrderRepository;
use Sylius\Component\Core\Model\ChannelInterface;
use Sylius\Component\Core\Model\CustomerInterface;
use Sylius\Component\Core\Model\OrderInterface;

class QuoteRepository extends OrderRepository implements QuoteRepositoryInterface
{
    public function findCartByChannel($id, ChannelInterface $channel): ?OrderInterface
    {
        return $this->createQueryBuilder('o')
            ->andWhere('o.id = :id')
            ->andWhere('o.state = :state')
            ->andWhere('o.channel = :channel')
            ->andWhere('o.quoteState = :quoteState')
            ->setParameter('id', $id)
            ->setParameter('state', OrderInterface::STATE_CART)
            ->setParameter('quoteState', QuoteInformationStateEnum::STATE_CART)
            ->setParameter('channel', $channel)
            ->getQuery()
            ->getOneOrNullResult()
        ;
    }

    public function findLatestNotEmptyQuoteCartByChannelAndCustomer(
        ChannelInterface $channel,
        CustomerInterface $customer,
    ): ?OrderInterface {
        return $this->createQueryBuilder('o')
            ->andWhere('o.state = :state')
            ->andWhere('o.quoteState = :quoteState')
            ->andWhere('o.channel = :channel')
            ->andWhere('o.customer = :customer')
            ->andWhere('o.items IS NOT EMPTY')
            ->andWhere('o.createdByGuest = :createdByGuest')
            ->setParameter('state', OrderInterface::STATE_CART)
            ->setParameter('quoteState', QuoteInformationStateEnum::STATE_CART)
            ->setParameter('channel', $channel)
            ->setParameter('customer', $customer)
            ->setParameter('createdByGuest', false)
            ->addOrderBy('o.createdAt', 'DESC')
            ->setMaxResults(1)
            ->getQuery()
            ->getOneOrNullResult()
        ;
    }

    public function findCartForSummary($id): ?OrderInterface
    {
        /** @var OrderInterface $order */
        $order = $this->createQueryBuilder('o')
            ->andWhere('o.id = :id')
            ->andWhere('o.state = :state')
            ->andWhere('o.quoteState = :quoteState')
            ->setParameter('id', $id)
            ->setParameter('state', OrderInterface::STATE_CART)
            ->setParameter('quoteState', QuoteInformationStateEnum::STATE_CART)
            ->getQuery()
            ->getOneOrNullResult()
        ;

        $this->associationHydrator->hydrateAssociations($order, [
            'adjustments',
            'items',
            'items.adjustments',
            'items.units',
            'items.units.adjustments',
            'items.variant',
            'items.variant.optionValues',
            'items.variant.optionValues.translations',
            'items.variant.product',
            'items.variant.product.translations',
            'items.variant.product.images',
            'items.variant.product.options',
            'items.variant.product.options.translations',
        ]);

        return $order;
    }

    public function findOneByNumberAndCustomer(string $number, CustomerInterface $customer): ?OrderInterface
    {
        return $this->createQueryBuilder('o')
            ->innerJoin('o.quoteInformation', 'qi')
            ->andWhere('o.customer = :customer')
            ->andWhere('qi.number = :number')
            ->setParameter('customer', $customer)
            ->setParameter('number', $number)
            ->getQuery()
            ->getOneOrNullResult()
        ;
    }

    public function findOneByNumber(string $number): ?OrderInterface
    {
        return $this->createQueryBuilder('o')
            ->innerJoin('o.quoteInformation', 'qi')
            ->andWhere('o.quoteState != :quoteState')
            ->andWhere('qi.number = :number')
            ->setParameter('number', $number)
            ->setParameter('quoteState', QuoteInformationStateEnum::STATE_CART)
            ->getQuery()
            ->getOneOrNullResult()
        ;
    }

    public function findExpiredQuotes(\DateTimeInterface $expirationDate): iterable
    {
        $queryBuilder = $this->createQueryBuilder('o');

        return $queryBuilder
            ->andWhere('o.expirationDate = :expirationDate')
            ->andWhere($queryBuilder->expr()->notIn('o.state', [QuoteInformationStateEnum::STATE_NEW, QuoteInformationStateEnum::STATE_EXPIRED]))
            ->setParameter('expirationDate', $expirationDate->format('Y-m-d'))
            ->getQuery()
            ->getResult()
        ;
    }
}
