<?php

declare(strict_types=1);

namespace Dedi\SyliusQuotePlugin\EventSubscriber;

use Dedi\SyliusQuotePlugin\Entity\OrderInterface;
use Dedi\SyliusQuotePlugin\Mailer\QuoteEmailManagerInterface;
use Dedi\SyliusQuotePlugin\NumberAssigner\QuoteNumberAssignerInterface;
use Dedi\SyliusQuotePlugin\Service\QuoteExpirationDateSetterInterface;
use Dedi\SyliusQuotePlugin\Service\QuoteIdSessionManagerInterface;
use Dedi\SyliusQuotePlugin\Service\QuoteItemLockerInterface;
use Dedi\SyliusQuotePlugin\Service\QuoteValidatedStateApplicatorInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\Workflow\Event\Event;

class QuoteWorkflowSubscriber implements EventSubscriberInterface
{
    public function __construct(
        private readonly QuoteValidatedStateApplicatorInterface $quoteValidatedStateApplicator,
        private readonly QuoteNumberAssignerInterface $quoteNumberAssigner,
        private readonly QuoteExpirationDateSetterInterface $quoteExpirationDateSetter,
        private readonly QuoteItemLockerInterface $quoteItemLocker,
        private readonly QuoteIdSessionManagerInterface $quoteIdSessionManager,
        private readonly QuoteEmailManagerInterface $quoteEmailManager,
        private readonly LoggerInterface $logger,
    ) {
    }

    public static function getSubscribedEvents(): array
    {
        return [
            'workflow.sylius_order_checkout.completed.complete' => 'afterCheckoutComplete',
            'workflow.dedi_sylius_quote_quote.transition.create' => 'beforeCreate',
            'workflow.dedi_sylius_quote_quote.transition.accept' => 'beforeAccept',
            'workflow.dedi_sylius_quote_quote.transition.accept_with_resupply' => 'beforeAccept',
            'workflow.dedi_sylius_quote_quote.completed.create' => 'afterCreate',
            'workflow.dedi_sylius_quote_quote.completed.accept' => 'afterAccept',
            'workflow.dedi_sylius_quote_quote.completed.accept_with_resupply' => 'afterAccept',
            'workflow.dedi_sylius_quote_quote.completed.deny' => 'afterDeny',
        ];
    }

    public function afterCheckoutComplete(Event $event): void
    {
        /** @var OrderInterface $order */
        $order = $event->getSubject();
        $this->quoteValidatedStateApplicator->apply($order);
    }

    public function beforeCreate(Event $event): void
    {
        /** @var OrderInterface $order */
        $order = $event->getSubject();
        $this->quoteNumberAssigner->assignNumber($order);
    }

    public function beforeAccept(Event $event): void
    {
        /** @var OrderInterface $order */
        $order = $event->getSubject();
        $this->quoteExpirationDateSetter->setExpirationDate($order);
    }

    public function afterCreate(Event $event): void
    {
        /** @var OrderInterface $order */
        $order = $event->getSubject();
        $this->quoteIdSessionManager->setQuoteIdInSession($order);

        try {
            $this->quoteEmailManager->sendConfirmationEmail($order);
        } catch (\Throwable $error) {
            $this->logger->error($error->getMessage());
        }
    }

    public function afterAccept(Event $event): void
    {
        /** @var OrderInterface $order */
        $order = $event->getSubject();

        $this->quoteItemLocker->lockItems($order);

        try {
            $this->quoteEmailManager->sendAcceptEmail($order);
        } catch (\Throwable $error) {
            $this->logger->error($error->getMessage());
        }
    }

    public function afterDeny(Event $event): void
    {
        /** @var OrderInterface $order */
        $order = $event->getSubject();

        try {
            $this->quoteEmailManager->sendDenyEmail($order);
        } catch (\Throwable $error) {
            $this->logger->error($error->getMessage());
        }
    }
}
