<?php

declare(strict_types=1);

namespace Dedi\SyliusQuotePlugin\EventListener\Sales;

use Dedi\SyliusQuotePlugin\DependencyInjection\Settings\SettingsInterface;
use Dedi\SyliusQuotePlugin\Entity\OrderInterface;
use Dedi\SyliusQuotePlugin\Entity\OrderItemInterface;
use Dedi\SyliusQuotePlugin\Service\QuoteConfigurationAwareInterface;
use Dedi\SyliusQuotePlugin\Specification\Order\IsFromQuoteSpecificationInterface as OrderIsFromQuoteSpecification;
use Dedi\SyliusQuotePlugin\Specification\OrderItem\IsFromQuoteSpecificationInterface as OrderItemIsFromQuoteSpecification;
use InvalidArgumentException;
use Sylius\Bundle\ResourceBundle\Event\ResourceControllerEvent;
use Sylius\Component\Order\Context\CartContextInterface;

class OrderItemQuoteLockListener implements QuoteConfigurationAwareInterface
{
    public function __construct(
        private readonly SettingsInterface $quoteConfiguration,
        private readonly CartContextInterface $cartContext,
        private readonly OrderIsFromQuoteSpecification $orderIsFromQuoteSpecification,
        private readonly OrderItemIsFromQuoteSpecification $orderItemIsFromQuoteSpecification,
    ) {
    }

    public function __invoke(ResourceControllerEvent $event)
    {
        if ($this->quoteConfiguration->isCartEditAllowed()) {
            return;
        }

        $orderItem = $event->getSubject();

        if (!$orderItem instanceof OrderItemInterface) {
            throw new InvalidArgumentException(sprintf(
                'Expected an instance of "%s", got "%s"',
                OrderItemInterface::class,
                get_class($orderItem),
            ));
        }

        /** @var ?OrderInterface $order */
        $order = $orderItem->getOrder();

        if (null !== $order && $this->orderItemIsFromQuoteSpecification->isSatisfiedBy($orderItem)) {
            $event->stop('dedi_sylius_quote.cart.cannot_update_from_quote');
        }

        if (null === $order) {
            $cart = $this->cartContext->getCart();
            if ($this->orderIsFromQuoteSpecification->isSatisfiedBy($cart)) {
                $event->stop('dedi_sylius_quote.cart.cannot_update_from_quote');
            }
        }
    }

    public function getQuoteConfiguration(): SettingsInterface
    {
        return $this->quoteConfiguration;
    }
}
