<?php

declare(strict_types=1);

namespace Dedi\SyliusOrderStatusHistoryPlugin\Workflow;

use Dedi\Contracts\SyliusPluginSettings\SettingsProviderInterface;
use Dedi\SyliusOrderStatusHistoryPlugin\Entity\OrderStatusHistoryInterface;
use Doctrine\ORM\EntityManagerInterface;
use Psr\Log\LoggerInterface;
use Sylius\Component\Core\Model\OrderInterface;
use Sylius\Component\Core\OrderPaymentTransitions;
use Sylius\Component\Core\OrderShippingTransitions;
use Sylius\Component\Order\OrderTransitions;
use Sylius\Component\Resource\Factory\FactoryInterface;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\Workflow\Event\CompletedEvent;
use Symfony\Component\Workflow\Transition;
use Symfony\Component\Workflow\WorkflowInterface;

class OrderWorkflowCallback
{
    public function __construct(
        private readonly SettingsProviderInterface $settingsProvider,
        private readonly FactoryInterface $orderStatusHistoryFactory,
        private readonly EntityManagerInterface $entityManager,
        private readonly RequestStack $requestStack,
        private readonly LoggerInterface $logger,
    ) {
    }

    public function onCompleted(CompletedEvent $event): void
    {
        /** @var OrderInterface $order */
        $order = $event->getSubject();
        $this->apply($order, $event->getTransition(), $event->getWorkflow());
    }

    public function apply(OrderInterface $order, Transition $transition, WorkflowInterface $stateMachine): void
    {
        try {
            $settings = $this->settingsProvider->provide();

            if (!$settings->isEnabled()) {
                return;
            }

            $request = $this->requestStack->getCurrentRequest();

            $stateMachineName = $stateMachine->getName();

            /** @var OrderStatusHistoryInterface $orderStatusHistory */
            $orderStatusHistory = $this->orderStatusHistoryFactory->createNew();
            $orderStatusHistory->setOrder($order);
            $orderStatusHistory->setStateMachineName($stateMachineName);
            $orderStatusHistory->setPreviousState($transition->getFroms()[0]);
            $orderStatusHistory->setTransition($transition->getTos()[0]);
            $orderStatusHistory->setUrl($request?->getUri());

            switch ($stateMachineName) {
                case OrderTransitions::GRAPH:
                    $orderStatusHistory->setState($order->getState());

                    break;
                case OrderPaymentTransitions::GRAPH:
                    $orderStatusHistory->setPaymentState($order->getPaymentState());

                    break;
                case OrderShippingTransitions::GRAPH:
                    $orderStatusHistory->setShippingState($order->getShippingState());

                    break;
            }

            $this->entityManager->persist($orderStatusHistory);
        } catch (\Throwable $exception) {
            $this->logger->error($exception->getMessage(), ['exception' => $exception]);
        }
    }
}
