<?php

declare(strict_types=1);

namespace Dedi\SyliusMegamenuPlugin\Controller;

use Dedi\SyliusMegamenuPlugin\Entity\EntryInterface;
use LogicException;
use Sylius\Bundle\ResourceBundle\Controller\ResourceController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

class EntryController extends ResourceController
{
    /**
     * Redirects the user to a configured route depending on the type of the entry.
     * Configuration example:
     *
     * dedi_megamenu_admin_entry_update:
     *     path: /menus/{menuId}/entry/{id}/edit
     *     defaults:
     *         _controller: dedi_megamenu.controller.entry:redirectAction
     *         _sylius:
     *             permission: true
     *             redirect:
     *                 routes:
     *                     product_catalog: dedi_megamenu_admin_product_catalog_entry_update
     *                     taxon_catalog: dedi_megamenu_admin_taxon_catalog_entry_update
     *                     direct_link: dedi_megamenu_admin_direct_link_entry_update
     *                 parameters:
     *                     id: $id
     *                     menuId: $menuId
     */
    public function redirectAction(
        Request $request,
    ): Response {
        $configuration = $this->requestConfigurationFactory->create($this->metadata, $request);

        /** @var EntryInterface $entry */
        $entry = $this->findOr404($configuration);
        $type = $entry->getType();

        // guess the route from the entry type
        $redirectConfig = $configuration->getParameters()->get('redirect');
        if (!is_array($redirectConfig) || !array_key_exists('routes', $redirectConfig)) {
            throw new LogicException('This route should be configured with a redirect and a list of routes.');
        }
        if (!is_array($redirectConfig['routes']) || !array_key_exists($type, $redirectConfig['routes'])) {
            throw new LogicException(sprintf(
                'routes should be an array and a route name should be configured as the value of the %s key',
                $type,
            ));
        }

        $redirectConfig = array_merge($redirectConfig, [
            'route' => $redirectConfig['routes'][$type],
        ]);
        $configuration->getParameters()->set('redirect', $redirectConfig);

        return $this->redirectHandler->redirectToResource($configuration, $entry);
    }

    public function moveUpAction(Request $request): Response
    {
        $configuration = $this->requestConfigurationFactory->create($this->metadata, $request);

        /** @var EntryInterface $entry */
        $entry = $this->findOr404($configuration);

        if ($entry->getPosition() > 0) {
            $entry->setPosition($entry->getPosition() - 1);
            $this->manager->flush();
        }

        return new JsonResponse('', Response::HTTP_NO_CONTENT);
    }

    public function moveDownAction(Request $request): Response
    {
        $configuration = $this->requestConfigurationFactory->create($this->metadata, $request);

        /** @var EntryInterface $entry */
        $entry = $this->findOr404($configuration);

        $entry->setPosition($entry->getPosition() + 1);
        $this->manager->flush();

        return new JsonResponse('', Response::HTTP_NO_CONTENT);
    }
}
