<?php

declare(strict_types=1);

namespace Dedi\SyliusMegamenuPlugin\DataFixtures\Factory;

use Behat\Transliterator\Transliterator;
use Dedi\SyliusMegamenuPlugin\Entity\DirectLinkEntry;
use Dedi\SyliusMegamenuPlugin\Entity\EntryInterface;
use Dedi\SyliusMegamenuPlugin\Entity\MenuInterface;
use Dedi\SyliusMegamenuPlugin\Entity\ProductCatalogEntry;
use Dedi\SyliusMegamenuPlugin\Entity\TaxonCatalogEntry;
use Sylius\Bundle\CoreBundle\Fixture\Factory\ExampleFactoryInterface;
use Sylius\Component\Locale\Model\LocaleInterface;
use Sylius\Component\Resource\Repository\RepositoryInterface;
use Sylius\Component\Taxonomy\Repository\TaxonRepositoryInterface;

class MegamenuEntryFactory implements ExampleFactoryInterface
{
    public function __construct(
        private readonly TaxonRepositoryInterface $taxonRepository,
        private readonly RepositoryInterface $localeRepository,
    ) {
    }

    public function create(array $options = []): EntryInterface
    {
        return match ($options['type']) {
            'direct_link' => $this->createDirectLink($options['menu'], $options),
            'taxon' => $this->createTaxon($options['menu'], $options),
            default => $this->createCatalog($options['menu'], $options),
        };
    }

    private function createDirectLink(MenuInterface $menu, array $options): EntryInterface
    {
        $directLinkEntry = new DirectLinkEntry();
        $code = Transliterator::transliterate($options['name']);
        $directLinkEntry->setCode($code);
        $directLinkEntry->setPosition($options['position']);
        $directLinkEntry->setMenu($menu);

        foreach ($this->getLocales() as $locale) {
            $directLinkEntry->setFallbackLocale($locale);
            $directLinkEntry->setCurrentLocale($locale);
            $directLinkEntry->setName($options['name']);
            $directLinkEntry->setLabel($options['label']);
            $directLinkEntry->setUrl($options['url']);
            $directLinkEntry->setIcon($options['icon']);
        }

        return $directLinkEntry;
    }

    private function createCatalog(MenuInterface $menu, array $options): EntryInterface
    {
        $productCatalogEntry = new ProductCatalogEntry();
        $code = Transliterator::transliterate($options['name']);
        $productCatalogEntry->setCode($code);
        $productCatalogEntry->setPosition($options['position']);
        $productCatalogEntry->setMenu($menu);

        foreach ($this->getLocales() as $locale) {
            $productCatalogEntry->setFallbackLocale($locale);
            $productCatalogEntry->setCurrentLocale($locale);
            $productCatalogEntry->setName($options['name']);
            $productCatalogEntry->setIcon($options['icon']);
            if (count($options['taxons']) === 0) {
                $taxons = $this->taxonRepository->findBy(['enabled' => true]);
                foreach ($taxons as $taxon) {
                    $productCatalogEntry->addTaxon($taxon);
                }
            }
            foreach ($options['taxons'] as $taxonCode) {
                $taxon = $this->taxonRepository->findOneBy(['code' => $taxonCode]);
                $productCatalogEntry->addTaxon($taxon);
            }
        }

        return $productCatalogEntry;
    }

    private function createTaxon(MenuInterface $menu, array $options): EntryInterface
    {
        $taxonCatalogEntry = new TaxonCatalogEntry();
        $code = Transliterator::transliterate($options['name']);
        $taxonCatalogEntry->setCode($code);
        $taxonCatalogEntry->setPosition($options['position']);
        $taxonCatalogEntry->setMenu($menu);

        $taxon = $this->taxonRepository->findOneBy(['code' => $options['taxon']]);

        foreach ($this->getLocales() as $locale) {
            $taxonCatalogEntry->setFallbackLocale($locale);
            $taxonCatalogEntry->setCurrentLocale($locale);
            $taxonCatalogEntry->setTaxon($taxon);
            $taxonCatalogEntry->setName($options['name']);
            $taxonCatalogEntry->setIcon($options['icon']);
        }

        return $taxonCatalogEntry;
    }

    private function getLocales(): iterable
    {
        /** @var LocaleInterface[] $locales */
        $locales = $this->localeRepository->findAll();
        foreach ($locales as $locale) {
            yield $locale->getCode();
        }
    }
}
