# Concepts

This document is an overview of the main concepts of the plugin. You will learn the following concepts:
* How a menu is displayed and built
* What are entries

## General

Menus are associated with a one to many relationship to entries.

Entries can have different types, in order to achieve this, the `Entry` entity is a mapped superclass configured with a single table inheritance mode.

Entries can be translated using the `EntryTranslationInterface`.

## Menu

Each menu is associated to a channel. The current menu for the current channel can be fetched using the `MenuContext::getMenu()` method.

For ease of use, the `MenuContext` is available in the twig global variable `dediMegamenu`.

Menus in the shop are rendered using the following template. It uses sylius template events in order to render tabs and content.

```twig
{# @DediSyliusMegamenuPlugin/Shop/Menu/_menu.html.twig #}

{% set menu = dediMegamenu.menu %}

{% if menu is not null %}
    <div data-menu>
        <div class="ui top attached tabular menu" data-items>
            {% for entry in menu.enabledEntries %}
                {{ sylius_template_event('dedi_megamenu.shop.menu.item.'~entry.type, {'entry': entry, 'index': loop.index}) }}
            {% endfor %}
        </div>
        {% for entry in menu.entries %}
            {{ sylius_template_event('dedi_megamenu.shop.menu.tab.'~entry.type, {'entry': entry, 'index': loop.index}) }}
        {% endfor %}
    </div>
{% endif %}
```

## Entries

The following is a guide on how to create a new type of entry. This is also useful to understand how entries work.

### Entities

As explained earlier, `Entry` is a mapped superclass. It is translatable using the `EntryTranslationInterface`.

In order to create your own entry type, you should create a new entity that extends `Entry` and implements `EntryInterface`.

Your new entity should be translated with a new entity implementing `EntryTranslationInterface`.

```php

use Dedi\SyliusMegamenuPlugin\Entity\Entry;
use Dedi\SyliusMegamenuPlugin\Entity\EntryInterface;
use Dedi\SyliusMegamenuPlugin\Entity\EntryTranslation;
use Dedi\SyliusMegamenuPlugin\Entity\EntryTranslationInterface;
use Sylius\Component\Resource\Model\TranslationInterface;

interface MyEntryInterface extends EntryInterface
{
    // ...
}

class TaxonCatalogEntry extends Entry implements MyEntryInterface
{
    // ...

    public function getType() : string
    {
        return 'my_entry';
    }
    
    protected function createTranslation() : TranslationInterface
    {
        return new MyEntryTranslation();
    }
}

interface MyEntryTranslationInterface extends EntryTranslationInterface, TranslationInterface
{
    // ...
}

class MyEntryTranslation extends EntryTranslation implements MyEntryTranslationInterface
{
    // ...
}

```
**This entity must be added to the discriminatory map of the `Entry` entity.**

For your new entity to be added to the discriminator map of Entry, you need to declare it as a service
and tag it with `dedi_megamenu_entry` such as follows :

```yaml
#config/services.yaml

App\NewEntity:
    tags:
        - { name: 'dedi_megamenu_entry' }
```

### Forms

Create forms for your new entities.

```php
use Dedi\SyliusMegamenuPlugin\Form\Type\EntryType;
use Dedi\SyliusMegamenuPlugin\Form\Type\EntryTranslationType;
use Sylius\Bundle\ResourceBundle\Form\Type\AbstractResourceType;
use Sylius\Bundle\ResourceBundle\Form\Type\ResourceTranslationsType;
use Symfony\Component\Form\FormBuilderInterface;

final class MyEntryType extends AbstractResourceType
{
    public function buildForm(FormBuilderInterface $builder, array $options)
    {
        $builder
            ->add('translations', ResourceTranslationsType::class, [
                'entry_type' => MyEntryTranslationType::class,
            ])
        ;
    }

    public function getParent(): string
    {
        return EntryType::class;
    }
}

final class MyEntryTranslationType extends AbstractResourceType
{
    public function buildForm(FormBuilderInterface $builder, array $options)
    {
        $builder
            // ...
        ;
    }

    public function getParent()
    {
        return EntryTranslationType::class;
    }
}
```

### Templates

Create templates for your form. You can use the `translationForm` macro from `@DediSyliusMegamenuPlugin/Admin/Entry/Common/translationForm.html.twig` in order to customize the translation form rendering.

For managing Taxon Type, and render only a kind of taxon, a twig global variable is usable. 
By default we allowed only taxon type = 'default'


```yaml
twig:
    globals:
        allowedTaxonType: ['default']
```
You can override it in `config/packages/twig.yaml` 
For example :
```yaml
twig:
    globals:
        allowedTaxonType: ['default', 'myCustomTaxonType']
```

```twig

{% from '@DediSyliusMegamenuPlugin/Admin/Entry/Common/translationForm.html.twig' import translationForm %}

{% include '@DediSyliusMegamenuPlugin/Admin/Entry/Common/_form.html.twig' %}

{# ... #}

{{ translationForm(form.translations, '../_translationForm.html.twig') }}
```

Override the `@DediSyliusMegamenuPlugin/Admin/Common/MenuTree/_createEntryButton.html.twig` to add your new entry so it can be created from the administration.

Override the `@DediSyliusMegamenuPlugin/Admin/Common/MenuTree/_icon.html.twig` to customize your entry icon.

### Routes

Create admin routes for your entry.

```yaml
app_megamenu_admin_my_entry_create:
    path: /menus/{menuId}/entry/new/my-entry
    methods: [GET, POST]
    defaults:
        _controller: app.controller.my_entry:createAction
        _sylius:
            section: admin
            permission: true
            factory:
                method: createForMenu
                arguments:
                    - expr:notFoundOnNull(service('dedi_megamenu.repository.menu').find($menuId))
            template: "@DediSyliusMegamenuPlugin/Admin/Entry/create.html.twig"
            vars:
                menu: expr:notFoundOnNull(service('dedi_megamenu.repository.menu').find($menuId))
                templates:
                    breadcrumb: "@DediSyliusMegamenuPlugin/Admin/Entry/Create/_breadcrumb.html.twig"
                    form: "@App/Admin/Entry/MyEntry/_form.html.twig"
                index:
                    route:
                        name: dedi_megamenu_admin_menu_show
                        parameters:
                            id: $menuId
                route:
                    parameters:
                        menuId: $menuId
                subheader: dedi_megamenu.ui.manage_your_entries
            redirect:
                route: app_megamenu_admin_my_entry_update
                parameters:
                    id: resource.id
                    menuId: $menuId

app_megamenu_admin_my_entry_update:
    path: /menus/{menuId}/entry/edit/{id}/my-entry
    methods: [GET, PUT]
    defaults:
        _controller: app.controller.my_entry:updateAction
        _sylius:
            section: admin
            permission: true
            template: "@DediSyliusMegamenuPlugin/Admin/Entry/update.html.twig"
            repository:
                method: findOneByIdAndMenuId
                arguments:
                    id: $id
                    menuId: $menuId
            vars:
                menu: expr:notFoundOnNull(service('dedi_megamenu.repository.menu').find($menuId))
                templates:
                    breadcrumb: "@DediSyliusMegamenuPlugin/Admin/Entry/Update/_breadcrumb.html.twig"
                    form: "@App/Admin/Entry/MyEntry/_form.html.twig"
                index:
                    route:
                        name: dedi_megamenu_admin_menu_show
                        parameters:
                            id: $menuId
                route:
                    parameters:
                        id: $id
                        menuId: $menuId
                subheader: dedi_megamenu.ui.manage_your_entries
            redirect:
                route: app_megamenu_admin_my_entry_update
                parameters:
                    menuId: $menuId
                    id: $id
```

### Menu rendering

Configure blocks on the following events depending on your needs.
* `dedi_megamenu.admin.menu.preview.item.my_entry` to render a menu item on the shop side for your entry
* `dedi_megamenu.admin.menu.preview.tab.my_entry` to render a menu tab on the shop side for your entry
* `dedi_megamenu.shop.menu.preview.item.my_entry` to render a menu item on the admin side for your entry
* `dedi_megamenu.shop.menu.preview.tab.my_entry` to render a menu tab on the admin side for your entry

### Sylius resource

Register you should configure your entity as a new sylius resource.
