<?php

declare(strict_types=1);

namespace Dedi\SyliusMarketingPlugin\Storage;

use Dedi\Contracts\SyliusPluginSettings\SettingsProviderInterface;
use Dedi\SyliusMarketingPlugin\DependencyInjection\Settings\SettingsInterface;
use Dedi\SyliusMarketingPlugin\Events\EventInterface;
use Dedi\SyliusMarketingPlugin\Specification\IsAsyncEvent;

class EventsStorage implements EventStorageInterface
{
    public const SYNC_SESSION_KEY = 'dedi_sylius_marketing_events_sync_events';

    public const ASYNC_SESSION_KEY = 'dedi_sylius_marketing_events_async_events';

    private SettingsProviderInterface $settingsProvider;

    private IsAsyncEvent $isAsyncEvent;

    private SessionManagerInterface $sessionManager;

    public function __construct(
        SettingsProviderInterface $settingsProvider,
        SessionManagerInterface $sessionManager,
        IsAsyncEvent $isAsyncEvent,
    ) {
        $this->settingsProvider = $settingsProvider;
        $this->sessionManager = $sessionManager;
        $this->isAsyncEvent = $isAsyncEvent;
    }

    public function store(EventInterface $event): void
    {
        /** @var SettingsInterface $settings */
        $settings = $this->settingsProvider->provide();
        $shouldReloadAsyncEvents = $settings->isAsyncReloaded();
        $isAsyncEvent = $this->isAsyncEvent->isSatisfiedBy($event);

        if ($isAsyncEvent) {
            $this->sessionManager->add(self::ASYNC_SESSION_KEY, [$event->getUniqId() => $event]);
        }

        if (!$isAsyncEvent || $shouldReloadAsyncEvents) {
            $this->sessionManager->add(self::SYNC_SESSION_KEY, [$event->getUniqId() => $event]);
        }
    }

    public function getSyncEvents(): array
    {
        return $this->sessionManager->get(self::SYNC_SESSION_KEY) ?? [];
    }

    public function getAsyncEvents(): array
    {
        return $this->sessionManager->get(self::ASYNC_SESSION_KEY) ?? [];
    }

    public function clear(): void
    {
        $this->sessionManager->remove(self::SYNC_SESSION_KEY);
        $this->sessionManager->remove(self::ASYNC_SESSION_KEY);
    }

    public function removeAsyncEvent(string $key): void
    {
        $events = $this->getAsyncEvents();
        unset($events[$key]);
        $this->sessionManager->set(self::ASYNC_SESSION_KEY, $events);
    }

    public function removeSyncEvent(string $key): void
    {
        $events = $this->getSyncEvents();
        unset($events[$key]);
        $this->sessionManager->set(self::SYNC_SESSION_KEY, $events);
    }
}
