<?php

declare(strict_types=1);

namespace Dedi\SyliusMarketingPlugin\Provider;

use Sylius\Component\Core\Model\ProductInterface;
use Sylius\Component\Core\Model\ProductVariantInterface;
use Sylius\Component\Locale\Context\LocaleContextInterface;
use Sylius\Component\Product\Model\Product;
use Sylius\Component\Product\Model\ProductOptionInterface;
use Sylius\Component\Taxonomy\Model\TaxonInterface;
use Sylius\Component\Taxonomy\Repository\TaxonRepositoryInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;

class TaxonProvider implements TaxonProviderInterface
{
    private TaxonRepositoryInterface $taxonRepository;

    private LocaleContextInterface $localeContext;

    private RequestStack $requestStack;

    public function __construct(
        TaxonRepositoryInterface $taxonRepository,
        LocaleContextInterface $localeContext,
        RequestStack $requestStack,
    ) {
        $this->taxonRepository = $taxonRepository;
        $this->localeContext = $localeContext;
        $this->requestStack = $requestStack;
    }

    public function getCategories(?ProductInterface $product): array
    {
        if (!$product instanceof ProductInterface) {
            return [];
        }

        $mainTaxon = $product->getMainTaxon();

        if (!$mainTaxon instanceof TaxonInterface) {
            return [];
        }

        $ancestors = $mainTaxon->getAncestors()->filter(function (TaxonInterface $taxon): bool {
            return $taxon->isEnabled();
        });

        $reverseAncestors = array_reverse($ancestors->toArray());

        $taxons = [];
        foreach ($reverseAncestors as $ancestor) {
            $taxons[] = $ancestor->getName();
        }
        $taxons[] = $mainTaxon->getName();

        return $taxons;
    }

    public function generateVariantOptions(ProductVariantInterface $variant, bool $isForCart): ?string
    {
        $itemVariant = '';
        $product = $variant->getProduct();

        if (!$product instanceof Product) {
            return null;
        }

        if ($isForCart) {
            $optionValues = $variant->getOptionValues();
        } else {
            $productOptions = $product->getOptions();
            $optionValues = [];

            /** @var ProductOptionInterface $option */
            foreach ($productOptions as $option) {
                $optionValues[] = $option->getValues()->first();
            }
        }

        if ($product->isConfigurable()) {
            foreach ($optionValues as $i => $optionValue) {
                if ($i < count($optionValues) - 1) {
                    $itemVariant .= $optionValue->getValue() . ' ';
                } else {
                    $itemVariant .= $optionValue->getValue();
                }
            }
        }

        return $itemVariant;
    }

    public function getListTaxonByCurrentUrl(): ?TaxonInterface
    {
        $request = $this->requestStack->getCurrentRequest();

        if (!$request instanceof Request) {
            return null;
        }

        $slug = $request->get('slug');

        return $this->getTaxonBySlug($slug);
    }

    public function getTaxonBySlug(string $slug): ?TaxonInterface
    {
        return $this->taxonRepository->findOneBySlug($slug, $this->localeContext->getLocaleCode());
    }
}
