<?php

declare(strict_types=1);

namespace Dedi\SyliusMarketingPlugin\Provider;

use Dedi\SyliusMarketingPlugin\Storage\SessionManagerInterface;
use Sylius\Component\Core\Model\ProductInterface;
use Sylius\Component\Locale\Context\LocaleContextInterface;
use Sylius\Component\Taxonomy\Model\TaxonInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;

class ItemListProvider implements ItemListProviderInterface
{
    private const HOMEPAGE_ITEM_LIST_NAME = 'Homepage';

    private TaxonProviderInterface $taxonProvider;

    private LocaleContextInterface $localeContext;

    private RequestStack $requestStack;

    private UrlGeneratorInterface $router;

    private SessionManagerInterface $sessionManager;

    public function __construct(
        TaxonProviderInterface $taxonProvider,
        LocaleContextInterface $localeContext,
        RequestStack $requestStack,
        UrlGeneratorInterface $router,
        SessionManagerInterface $sessionManager,
    ) {
        $this->taxonProvider = $taxonProvider;
        $this->localeContext = $localeContext;
        $this->requestStack = $requestStack;
        $this->router = $router;
        $this->sessionManager = $sessionManager;
    }

    public function getItemListIdForProduct(ProductInterface $product): ?string
    {
        $request = $this->requestStack->getCurrentRequest();

        $referer = null;
        if ($request instanceof Request) {
            $referer = $request->headers->get('referer');
        }

        if (null !== $referer && $this->isHomepageReferer()) {
            return null;
        }

        $lastSeenTaxon = $this->sessionManager->get(SessionManagerInterface::LAST_TAXON_KEY_SUFFIX);

        $taxon = null;
        if (isset($lastSeenTaxon['slug'])) {
            $taxon = $this->taxonProvider->getTaxonBySlug($lastSeenTaxon['slug']);
        }

        if ($taxon instanceof TaxonInterface) {
            $this->sessionManager->add($product->getId() . SessionManagerInterface::ITEM_LIST_KEY_SUFFIX, [
                'item_list_id' => $taxon->getId(),
            ]);

            return (string) $taxon->getId();
        }

        return null;
    }

    public function getItemListNameForProduct(ProductInterface $product): ?string
    {
        $request = $this->requestStack->getCurrentRequest();

        $referer = null;
        if ($request instanceof Request) {
            $referer = $request->headers->get('referer');
        }

        if (null !== $referer && $this->isHomepageReferer()) {
            $this->sessionManager->add($product->getId() . SessionManagerInterface::ITEM_LIST_KEY_SUFFIX, [
                'item_list_name' => self::HOMEPAGE_ITEM_LIST_NAME,
            ]);

            return self::HOMEPAGE_ITEM_LIST_NAME;
        }

        $lastSeenTaxon = $this->sessionManager->get(SessionManagerInterface::LAST_TAXON_KEY_SUFFIX);

        $taxon = null;
        if (isset($lastSeenTaxon['slug'])) {
            $taxon = $this->taxonProvider->getTaxonBySlug($lastSeenTaxon['slug']);
        }

        $itemListName = null;
        if ($taxon instanceof TaxonInterface) {
            $itemListName = $taxon->getName();
        }

        $this->sessionManager->add($product->getId() . SessionManagerInterface::ITEM_LIST_KEY_SUFFIX, [
            'item_list_name' => $itemListName,
        ]);

        return $itemListName;
    }

    public function getItemListIdForTaxon(): ?string
    {
        $taxon = $this->taxonProvider->getListTaxonByCurrentUrl();

        if ($taxon instanceof TaxonInterface) {
            return (string) $taxon->getId();
        }

        return null;
    }

    public function getItemListNameForTaxon(): ?string
    {
        $taxon = $this->taxonProvider->getListTaxonByCurrentUrl();

        if ($taxon instanceof TaxonInterface) {
            return $taxon->getName();
        }

        return null;
    }

    private function isHomepageReferer(): bool
    {
        $request = $this->requestStack->getCurrentRequest();

        if (!$request instanceof Request) {
            return false;
        }

        $referer = $request->headers->get('referer');

        if (null === $referer) {
            return false;
        }

        return $referer === $this->router->generate(
            'sylius_shop_homepage',
            ['_locale' => $this->localeContext->getLocaleCode()],
            UrlGeneratorInterface::ABSOLUTE_URL,
        );
    }
}
