<?php

declare(strict_types=1);

namespace Dedi\SyliusMarketingPlugin\Factory;

use Dedi\SyliusMarketingPlugin\Item\Item;
use Dedi\SyliusMarketingPlugin\Item\ItemInterface;
use Dedi\SyliusMarketingPlugin\Provider\ItemListProviderInterface;
use Dedi\SyliusMarketingPlugin\Provider\TaxonProviderInterface;
use Dedi\SyliusMarketingPlugin\Storage\SessionManagerInterface;
use Sylius\Component\Channel\Context\ChannelContextInterface;
use Sylius\Component\Core\Model\Channel;
use Sylius\Component\Core\Model\ChannelPricingInterface;
use Sylius\Component\Core\Model\OrderInterface;
use Sylius\Component\Core\Model\OrderItemInterface;
use Sylius\Component\Core\Model\Product;
use Sylius\Component\Core\Model\ProductInterface;
use Sylius\Component\Core\Model\ProductVariantInterface;
use Sylius\Component\Core\Model\PromotionCouponInterface;
use Sylius\Component\Currency\Context\CurrencyContextInterface;
use Sylius\Component\Product\Resolver\ProductVariantResolverInterface;

class ItemFactory implements ItemFactoryInterface
{
    private ChannelContextInterface $channelContext;

    private CurrencyContextInterface $currencyContext;

    private ProductVariantResolverInterface $productVariantResolver;

    private TaxonProviderInterface $taxonProvider;

    private ItemListProviderInterface $itemListProvider;

    private SessionManagerInterface $sessionManager;

    public function __construct(
        ChannelContextInterface $channelContext,
        CurrencyContextInterface $currencyContext,
        ProductVariantResolverInterface $productVariantResolver,
        TaxonProviderInterface $taxonProvider,
        ItemListProviderInterface $itemListProvider,
        SessionManagerInterface $sessionManager,
    ) {
        $this->channelContext = $channelContext;
        $this->currencyContext = $currencyContext;
        $this->productVariantResolver = $productVariantResolver;
        $this->taxonProvider = $taxonProvider;
        $this->itemListProvider = $itemListProvider;
        $this->sessionManager = $sessionManager;
    }

    public function fromProductVariant(
        ProductVariantInterface $variant,
        bool $isForCart = false,
        ?int $position = null,
        string $itemListId = null,
        string $itemListName = null,
    ): ItemInterface {
        /** @var Channel $channel */
        $channel = $this->channelContext->getChannel();
        $pricing = $variant->getChannelPricingForChannel($channel);

        $price = 0;
        if ($pricing instanceof ChannelPricingInterface) {
            $price = $pricing->getPrice();
        }

        if (!is_int($price)) {
            $price = $pricing->getOriginalPrice();
        }

        /** @var Product|null $product */
        $product = $variant->getProduct();

        if (null === $product) {
            return $this->returnMissingItem('product missing for variant: ' . $variant->getId());
        }

        $itemVariant = $this->taxonProvider->generateVariantOptions($variant, $isForCart);

        if (null === $itemListId) {
            $itemListId = $this->itemListProvider->getItemListIdForProduct($product) ?? '';
        }
        if (null === $itemListName) {
            $itemListName = $this->itemListProvider->getItemListNameForProduct($product) ?? '';
        }

        $item = new Item();
        $item->setItemId((string) $product->getId());
        $item->setItemName((string) $product->getName());
        $item->setPrice(round($price / 100, 2));
        $item->setCurrency($this->currencyContext->getCurrencyCode());

        $categories = $this->taxonProvider->getCategories($product);

        $item->setItemCategory(count($categories) > 0 ? (string) array_shift($categories) : '');
        $item->setItemCategory2(count($categories) > 0 ? (string) array_shift($categories) : '');
        $item->setItemCategory3(count($categories) > 0 ? (string) array_shift($categories) : '');

        $item->setItemListId($itemListId);
        $item->setItemListName($itemListName);
        $item->setItemVariant($itemVariant);
        $item->setLocationId($channel->getName());

        return $item;
    }

    public function fromProduct(
        ProductInterface $product,
        ?int $position = null,
        string $itemListId = null,
        string $itemListName = null,
    ): ItemInterface {
        $variants = $product->getEnabledVariants();

        if ($variants->isEmpty()) {
            return $this->returnMissingItem('variant missing for product ' . $product->getId());
        }

        /** @var ProductVariantInterface $variant */
        $variant = $this->productVariantResolver->getVariant($product);

        return $this->fromProductVariant($variant, false, $position, $itemListId, $itemListName);
    }

    public function fromOrderItem(OrderItemInterface $orderItem): ItemInterface
    {
        $variant = $orderItem->getVariant();

        if (!$variant instanceof ProductVariantInterface) {
            return $this->returnMissingItem('variant missing for orderItem: ' . $orderItem->getId());
        }

        $product = $variant->getProduct();
        $itemListData = $this->sessionManager->get($product->getId() . SessionManagerInterface::ITEM_LIST_KEY_SUFFIX);

        $itemListId = isset($itemListData['item_list_id']) ? (string) $itemListData['item_list_id'] : null;
        $itemListName = $itemListData['item_list_name'] ?? null;

        $item = $this->fromProductVariant(
            $variant,
            true,
            null,
            $itemListId,
            $itemListName,
        );
        $price = ($orderItem->getUnitPrice() > 0) ? $orderItem->getUnitPrice() / 100 : (float) $item->getPrice();

        return $item
            ->setQuantity($orderItem->getQuantity())
            ->setPrice((float) $price)
            ->setDiscount(($orderItem->getUnitPrice() - $orderItem->getFullDiscountedUnitPrice()) / 100)
            ->setCoupon($this->getCoupon($orderItem))
        ;
    }

    private function getCoupon(OrderItemInterface $orderItem): ?string
    {
        $order = $orderItem->getOrder();
        if (!$order instanceof OrderInterface) {
            return null;
        }

        $code = null;
        if ($order->getPromotionCoupon() instanceof PromotionCouponInterface) {
            $code = $order->getPromotionCoupon()->getCode();
        }

        return $code;
    }

    private function returnMissingItem(string $message): ItemInterface
    {
        $item = new Item();
        $item->setItemId('0');
        $item->setItemName($message);
        $item->setPrice(0);
        $item->setCurrency($this->currencyContext->getCurrencyCode());

        return $item;
    }
}
