<?php

declare(strict_types=1);

namespace Dedi\SyliusMarketingPlugin\Factory\Events;

use Dedi\SyliusMarketingPlugin\ControllerEvent\SelectItemEvent;
use Dedi\SyliusMarketingPlugin\Events\ItemsContainerInterface;
use Dedi\SyliusMarketingPlugin\Events\Search;
use Dedi\SyliusMarketingPlugin\Events\SelectItem;
use Dedi\SyliusMarketingPlugin\Events\ViewItem;
use Dedi\SyliusMarketingPlugin\Events\ViewItemList;
use Dedi\SyliusMarketingPlugin\Factory\ItemFactoryInterface;
use Dedi\SyliusMarketingPlugin\Provider\ItemListProviderInterface;
use Dedi\SyliusMarketingPlugin\Storage\TaxonNavigationHistoryManagerInterface;
use Sylius\Component\Core\Model\ProductInterface;
use Sylius\Component\Product\Repository\ProductRepositoryInterface;

class NavigationEventFactory implements NavigationEventFactoryInterface
{
    private ItemFactoryInterface $itemFactory;

    private ProductRepositoryInterface $productRepository;

    private ItemListProviderInterface $itemListProvider;

    private TaxonNavigationHistoryManagerInterface $taxonNavigationHistoryManager;

    public function __construct(
        ItemFactoryInterface $itemFactory,
        ProductRepositoryInterface $productRepository,
        ItemListProviderInterface $itemListProvider,
        TaxonNavigationHistoryManagerInterface $taxonNavigationHistoryManager,
    ) {
        $this->itemFactory = $itemFactory;
        $this->productRepository = $productRepository;
        $this->itemListProvider = $itemListProvider;
        $this->taxonNavigationHistoryManager = $taxonNavigationHistoryManager;
    }

    public function search(string $searchTerm): Search
    {
        return new Search($searchTerm);
    }

    public function viewItemList(array $products = []): ViewItemList
    {
        $this->taxonNavigationHistoryManager->storeLastSeenTaxon();

        $itemListId = $this->itemListProvider->getItemListIdForTaxon();
        $itemListName = $this->itemListProvider->getItemListNameForTaxon();

        $viewItemList = (new ViewItemList())
            ->setItemListId($itemListId)
            ->setItemListName($itemListName)
        ;
        $this->addItems($viewItemList, $products, $itemListId, $itemListName);

        return $viewItemList;
    }

    public function viewItem(ProductInterface $product): ViewItem
    {
        $viewItem = new ViewItem();
        $viewItem->addItem($this->itemFactory->fromProduct($product));

        return $viewItem;
    }

    public function selectItem(SelectItemEvent $selectItemEvent): SelectItem
    {
        $selectItem = new SelectItem();

        $product = $this->productRepository->find($selectItemEvent->productId);

        if (!$product instanceof ProductInterface) {
            return $selectItem;
        }

        $selectItem->addItem($this->itemFactory->fromProduct($product, $selectItemEvent->productPosition));

        return $selectItem;
    }

    private function addItems(
        ItemsContainerInterface $event,
        array $products,
        ?string $itemListId,
        ?string $itemListName,
    ): void {
        foreach ($products as $product) {
            $event->addItem(
                $this->itemFactory->fromProduct($product, null, $itemListId, $itemListName),
            );
        }
    }
}
