<?php

declare(strict_types=1);

namespace Dedi\SyliusMarketingPlugin\Factory\Events;

use Dedi\SyliusMarketingPlugin\Events\AddAddressInfo;
use Dedi\SyliusMarketingPlugin\Events\AddPaymentInfo;
use Dedi\SyliusMarketingPlugin\Events\AddShippingInfo;
use Dedi\SyliusMarketingPlugin\Events\BeginCheckout;
use Dedi\SyliusMarketingPlugin\Events\ItemsContainerInterface;
use Dedi\SyliusMarketingPlugin\Events\Purchase;
use Dedi\SyliusMarketingPlugin\Factory\ItemFactoryInterface;
use Doctrine\Common\Collections\Collection;
use Sylius\Component\Core\Model\Order;
use Sylius\Component\Core\Model\OrderInterface;
use Sylius\Component\Core\Model\PaymentInterface;
use Sylius\Component\Core\Model\PaymentMethodInterface;
use Sylius\Component\Core\Model\PromotionCouponInterface;
use Sylius\Component\Core\Model\ShipmentInterface;
use Sylius\Component\Core\Model\ShippingMethodInterface;
use Sylius\Component\Order\Context\CartContextInterface;

class CheckoutEventFactory implements CheckoutEventFactoryInterface
{
    private CartContextInterface $cartContext;

    private ItemFactoryInterface $itemFactory;

    public function __construct(CartContextInterface $cartContext, ItemFactoryInterface $itemFactory)
    {
        $this->cartContext = $cartContext;
        $this->itemFactory = $itemFactory;
    }

    public function beginCheckout(): BeginCheckout
    {
        /** @var OrderInterface $order */
        $order = $this->cartContext->getCart();

        $beginCheckout = new BeginCheckout();
        $beginCheckout->setCoupon($this->getCoupon($order));

        $this->addItems($beginCheckout, $order->getItems());

        return $beginCheckout;
    }

    public function addAddressInfo(): AddAddressInfo
    {
        /** @var OrderInterface $order */
        $order = $this->cartContext->getCart();

        $addAddress = new AddAddressInfo();
        $addAddress->setCoupon($this->getCoupon($order));

        $this->addItems($addAddress, $order->getItems());

        return $addAddress;
    }

    public function addPaymentInfo(): AddPaymentInfo
    {
        /** @var OrderInterface $order */
        $order = $this->cartContext->getCart();
        $lastPayment = $order->getLastPayment();

        $addPaymentInfo = new AddPaymentInfo();
        $addPaymentInfo->setCoupon($this->getCoupon($order));

        $paymentMethod = null;
        if ($lastPayment instanceof PaymentInterface) {
            $paymentMethod = $lastPayment->getMethod();
        }

        $paymentMethodName = null;
        if ($paymentMethod instanceof PaymentMethodInterface) {
            $paymentMethodName = $paymentMethod->getName();
        }

        $addPaymentInfo->setPaymentType($paymentMethodName);

        $this->addItems($addPaymentInfo, $order->getItems());

        $addPaymentInfo->setTotal($order->getTotal() / 100);

        return $addPaymentInfo;
    }

    public function addShippingInfo(): AddShippingInfo
    {
        /** @var OrderInterface $order */
        $order = $this->cartContext->getCart();
        /** @var ShipmentInterface|null $shipment */
        $shipment = $order->getShipments()->first();

        $addShippingInfo = new AddShippingInfo();
        $addShippingInfo->setCoupon($this->getCoupon($order));

        $shippingMethod = null;
        if ($shipment instanceof ShipmentInterface) {
            $shippingMethod = $shipment->getMethod();
        }

        $shippingMethodName = null;
        if ($shippingMethod instanceof ShippingMethodInterface) {
            $shippingMethodName = $shippingMethod->getName();
        }

        $addShippingInfo->setShippingTier($shippingMethodName);

        $this->addItems($addShippingInfo, $order->getItems());

        $addShippingInfo->setTotal($order->getTotal() / 100);

        return $addShippingInfo;
    }

    public function purchase(Order $order): Purchase
    {
        $purchase = new Purchase((string) $order->getNumber());

        $purchase->setCoupon($this->getCoupon($order));
        $purchase->setShipping($order->getShippingTotal() / 100);
        $purchase->setTax($order->getTaxTotal() / 100);

        $items = $order->getItems();
        $this->addItems($purchase, $items);

        $purchase->setTotal($order->getTotal() / 100);

        return $purchase;
    }

    private function addItems(ItemsContainerInterface $event, Collection $items): void
    {
        foreach ($items as $orderItem) {
            $event->addItem(
                $this->itemFactory->fromOrderItem($orderItem),
            );
        }
    }

    private function getCoupon(OrderInterface $order): ?string
    {
        $code = null;
        if ($order->getPromotionCoupon() instanceof PromotionCouponInterface) {
            $code = $order->getPromotionCoupon()->getCode();
        }

        return $code;
    }
}
