<?php

declare(strict_types=1);

namespace Dedi\SyliusMarketingPlugin\EventListener;

use Dedi\SyliusMarketingPlugin\ControllerEvent\SelectItemEvent;
use Dedi\SyliusMarketingPlugin\Extractor\RequestProductExtractorInterface;
use Dedi\SyliusMarketingPlugin\Factory\Events\NavigationEventFactoryInterface;
use Dedi\SyliusMarketingPlugin\Storage\EventsStorage;
use Sylius\Bundle\ResourceBundle\Event\ResourceControllerEvent;
use Sylius\Component\Core\Model\ProductInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpKernel\Event\ControllerArgumentsEvent;
use Symfony\Contracts\EventDispatcher\Event;

class NavigationEventListener extends MarketingListener implements EventSubscriberInterface
{
    public const TAXONS_ROUTES = ['sylius_shop_product_index', 'sylius_shop_product_paginate'];

    private NavigationEventFactoryInterface $navEventFactory;

    private EventsStorage $eventsStorage;

    private RequestProductExtractorInterface $requestProductExtractor;

    public function __construct(
        NavigationEventFactoryInterface $navEventFactory,
        EventsStorage $eventsStorage,
        RequestProductExtractorInterface $requestProductExtractor,
    ) {
        $this->navEventFactory = $navEventFactory;
        $this->eventsStorage = $eventsStorage;
        $this->requestProductExtractor = $requestProductExtractor;
    }

    public static function getSubscribedEvents(): array
    {
        return [
            'sylius.product.show' => 'viewItem',
            'controller_select_item' => 'selectItem',
            'kernel.controller_arguments' => 'onKernelRequest',
        ];
    }

    public function onKernelRequest(ControllerArgumentsEvent $event): void
    {
        $routeName = $event->getRequest()->attributes->get('_route');

        if (!in_array($routeName, self::TAXONS_ROUTES, true) ||
            !$event->isMainRequest() ||
            !\is_array($event->getController())
        ) {
            return;
        }

        if (!$this->isEnabled()) {
            return;
        }

        $searchTerm = $this->getSearchTerm($event->getRequest());
        if (null !== $searchTerm) {
            $this->eventsStorage->store(
                $this->navEventFactory->search($searchTerm),
            );
        }

        $this->viewItemList($event);
    }

    public function viewItem(ResourceControllerEvent $resourceControllerEvent): void
    {
        $product = $resourceControllerEvent->getSubject();

        if (!$product instanceof ProductInterface) {
            return;
        }

        if (!$this->isEnabled()) {
            return;
        }

        $this->eventsStorage->store(
            $this->navEventFactory->viewItem($product),
        );
    }

    public function viewItemList(Event $event): void
    {
        $products = $this->requestProductExtractor->extract($event);

        $this->eventsStorage->store(
            $this->navEventFactory->viewItemList($products),
        );
    }

    public function selectItem(SelectItemEvent $selectItemEvent): void
    {
        if (!$this->isEnabled()) {
            return;
        }

        $this->eventsStorage->store(
            $this->navEventFactory->selectItem($selectItemEvent),
        );
    }

    private function getSearchTerm(Request $request): ?string
    {
        $criteria = $request->query->all('criteria');

        if (isset($criteria['search']['value'])) {
            return (string) $criteria['search']['value'];
        }

        return null;
    }
}
