<?php

declare(strict_types=1);

namespace Dedi\SyliusMarketingPlugin\EventListener;

use Dedi\SyliusMarketingPlugin\Factory\Events\CheckoutEventFactoryInterface;
use Dedi\SyliusMarketingPlugin\Storage\EventsStorage;
use Sylius\Bundle\ResourceBundle\Event\ResourceControllerEvent;
use Sylius\Component\Core\Model\Order;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\EventDispatcher\GenericEvent;
use Symfony\Component\HttpKernel\Event\ControllerArgumentsEvent;

class CheckoutEventListener extends MarketingListener implements EventSubscriberInterface
{
    private CheckoutEventFactoryInterface $checkoutEventFactory;

    private EventsStorage $eventsStorage;

    private string $beginCheckoutRouteName = 'sylius_shop_checkout_address';

    public function __construct(
        CheckoutEventFactoryInterface $checkoutEventFactory,
        EventsStorage $eventsStorage,
        string $beginCheckoutRouteName = 'sylius_shop_checkout_address',
    ) {
        $this->checkoutEventFactory = $checkoutEventFactory;
        $this->eventsStorage = $eventsStorage;
        $this->beginCheckoutRouteName = $beginCheckoutRouteName;
    }

    public static function getSubscribedEvents(): array
    {
        return [
            'kernel.controller_arguments' => 'onKernelRequest', //beginCheckout
            'sylius.order.post_address' => 'addAddressInfo',
            'sylius.order.post_payment' => 'addPaymentInfo',
            'sylius.order.post_select_payment' => 'addPaymentInfo',
            'sylius.order.post_select_shipping' => 'addShippingInfo',
            'sylius.order.post_admin_create' => 'purchase',
            'sylius.order.post_complete' => 'purchase',
        ];
    }

    public function onKernelRequest(ControllerArgumentsEvent $event): void
    {
        $routeName = $event->getRequest()->attributes->get('_route');

        if ($this->beginCheckoutRouteName !== $routeName ||
            !$event->isMainRequest() ||
            !\is_array($event->getController()) ||
            $event->getRequest()->getMethod() !== 'GET'
        ) {
            return;
        }

        if (!$this->isEnabled()) {
            return;
        }

        $this->eventsStorage->store(
            $this->checkoutEventFactory->beginCheckout(),
        );
    }

    public function addAddressInfo(GenericEvent $event): void
    {
        if (!$this->isEnabled()) {
            return;
        }

        $this->eventsStorage->store(
            $this->checkoutEventFactory->addAddressInfo(),
        );
    }

    public function addPaymentInfo(GenericEvent $event): void
    {
        if (!$this->isEnabled()) {
            return;
        }

        $this->eventsStorage->store(
            $this->checkoutEventFactory->addPaymentInfo(),
        );
    }

    public function addShippingInfo(GenericEvent $event): void
    {
        if (!$this->isEnabled()) {
            return;
        }

        $this->eventsStorage->store(
            $this->checkoutEventFactory->addShippingInfo(),
        );
    }

    public function purchase(ResourceControllerEvent $resourceControllerEvent): void
    {
        if (!$this->isEnabled()) {
            return;
        }

        /** @var Order $order */
        $order = $resourceControllerEvent->getSubject();

        $this->eventsStorage->store(
            $this->checkoutEventFactory->purchase($order),
        );
    }
}
