<?php

declare(strict_types=1);

namespace Dedi\SyliusMarketingPlugin\EventListener;

use Dedi\SyliusMarketingPlugin\Factory\Events\CartEventFactoryInterface;
use Dedi\SyliusMarketingPlugin\Storage\EventsStorage;
use Sylius\Bundle\ResourceBundle\Event\ResourceControllerEvent;
use Sylius\Component\Core\Model\OrderItemInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpKernel\Event\ControllerArgumentsEvent;

class CartEventListener extends MarketingListener implements EventSubscriberInterface
{
    private CartEventFactoryInterface $cartEventFactory;

    private EventsStorage $eventsStorage;

    private string $viewCartRouteName = 'sylius_shop_cart_summary';

    public function __construct(
        CartEventFactoryInterface $cartEventFactory,
        EventsStorage $eventsStorage,
        string $viewCartRouteName = 'sylius_shop_cart_summary',
    ) {
        $this->cartEventFactory = $cartEventFactory;
        $this->eventsStorage = $eventsStorage;
        $this->viewCartRouteName = $viewCartRouteName;
    }

    public static function getSubscribedEvents(): array
    {
        return [
            'sylius.order_item.pre_add' => 'addToCart',
            'kernel.controller_arguments' => 'onKernelRequest', //viewCart
            'sylius.order_item.post_remove' => 'removeFromCart',
        ];
    }

    public function addToCart(ResourceControllerEvent $resourceControllerEvent): void
    {
        $orderItem = $resourceControllerEvent->getSubject();

        if (!$orderItem instanceof OrderItemInterface) {
            return;
        }

        if (!$this->isEnabled()) {
            return;
        }

        $this->eventsStorage->store(
            $this->cartEventFactory->addToCart($orderItem),
        );
    }

    public function onKernelRequest(ControllerArgumentsEvent $event): void
    {
        $routeName = (string) ($event->getRequest()->attributes->get('_route'));

        if ($this->viewCartRouteName !== $routeName ||
            !$event->isMainRequest() ||
            !\is_array($event->getController())
        ) {
            return;
        }

        if (!$this->isEnabled()) {
            return;
        }

        $this->eventsStorage->store(
            $this->cartEventFactory->viewCart(),
        );
    }

    public function removeFromCart(ResourceControllerEvent $resourceControllerEvent): void
    {
        $orderItem = $resourceControllerEvent->getSubject();

        if (!$orderItem instanceof OrderItemInterface) {
            return;
        }

        if (!$this->isEnabled()) {
            return;
        }

        $this->eventsStorage->store(
            $this->cartEventFactory->removeFromCart($orderItem),
        );
    }
}
