```php
<?php

declare(strict_types=1);

namespace App\Marketing\Datalayer\Extractor;

use App\Service\Catalog\Search\Handler\ShopProductListDataHandler;
use BitBag\SyliusElasticsearchPlugin\Controller\RequestDataHandler\PaginationDataHandlerInterface;
use BitBag\SyliusElasticsearchPlugin\Controller\RequestDataHandler\SortDataHandlerInterface;
use BitBag\SyliusElasticsearchPlugin\Finder\ShopProductsFinderInterface;
use BitBag\SyliusElasticsearchPlugin\Form\Type\ShopProductsFilterType;
use Dedi\SyliusMarketingPlugin\Extractor\RequestProductExtractorInterface;
use Symfony\Component\Form\FormFactoryInterface;
use Symfony\Component\Form\FormInterface;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\PropertyAccess\PropertyAccess;
use Symfony\Contracts\EventDispatcher\Event;

class RequestProductBitbagExtractor implements RequestProductExtractorInterface
{
    public function __construct(
        private readonly FormFactoryInterface $formFactory,
        private readonly ShopProductListDataHandler $shopProductListDataHandler,
        private readonly SortDataHandlerInterface $shopProductsSortDataHandler,
        private readonly PaginationDataHandlerInterface $paginationDataHandler,
        private readonly ShopProductsFinderInterface $shopProductsFinder,
        private readonly RequestStack $requestStack,
    ) {
    }

    public function extract(Event $event): array
    {
        $request = $this->requestStack->getCurrentRequest();

        $form = $this->formFactory->create(ShopProductsFilterType::class);
        $form->handleRequest($request);
        $requestData = array_merge(
            $form->getData(),
            $request->query->all(),
            ['slug' => $request->get('slug')],
        );

        if (!$form->isValid()) {
            $requestData = $this->clearInvalidEntries($form, $requestData);
        }

        $data = array_merge(
            $this->shopProductListDataHandler->retrieveData($requestData),
            $this->shopProductsSortDataHandler->retrieveData($requestData),
            $this->paginationDataHandler->retrieveData($requestData),
        );

        $pager = $this->shopProductsFinder->find($data);

        $products = [];
        foreach ($pager->getIterator() as $product) {
            $products[] = $product;
        }

        return $products;
    }

    private function clearInvalidEntries(FormInterface $form, array $requestData): array
    {
        $propertyAccessor = PropertyAccess::createPropertyAccessor();
        foreach ($form->getErrors(true) as $error) {
            $errorOrigin = $error->getOrigin();
            $propertyAccessor->setValue(
                $requestData,
                ($errorOrigin->getParent()->getPropertyPath() ?? '') . $errorOrigin->getPropertyPath(),
                '',
            );
        }

        return $requestData;
    }
}
```