<?php

declare(strict_types=1);

namespace Dedi\SyliusMaintenancePlugin\EventSubscriber;

use Dedi\Contracts\SyliusPluginSettings\SettingsProviderInterface;
use Dedi\SyliusMaintenancePlugin\DependencyInjection\Settings\SettingsInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Session\Flash\FlashBagInterface;
use Symfony\Component\HttpKernel\Event\RequestEvent;
use Symfony\Component\HttpKernel\HttpKernelInterface;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Symfony\Contracts\Translation\TranslatorInterface;

class MaintenanceEventSubscriber implements EventSubscriberInterface
{
    public function __construct(
        private readonly UrlGeneratorInterface $urlGenerator,
        private readonly bool $envMaintenanceActive,
        private readonly string $envIpList,
        private readonly SettingsProviderInterface $settingsProvider,
        private readonly string $adminPrefix,
        private readonly TranslatorInterface $translator,
    ) {
    }

    public static function getSubscribedEvents(): array
    {
        return [
            RequestEvent::class => 'handle',
        ];
    }

    public function handle(RequestEvent $event): void
    {
        /** @var SettingsInterface $settings */
        $settings = $this->settingsProvider->provide();
        if (!$settings->isEnabled()) {
            return;
        }

        $request = $event->getRequest();
        $routeName = $request->get('_route', '');

        $isMaintenance = $this->isMaintenance($event, $routeName);

        if ($isMaintenance && !$this->isRouteMaintenance($routeName)) {
            $event->setResponse(new RedirectResponse($this->urlGenerator->generate('dedi_maintenance_shop_maintenance')));

            return;
        }

        if (!$isMaintenance && $this->isRouteMaintenance($routeName)) {
            $event->setResponse(new RedirectResponse($this->urlGenerator->generate('sylius_shop_homepage')));
        }
    }

    private function getIpList(): array
    {
        /** @var SettingsInterface $settings */
        $settings = $this->settingsProvider->provide();

        $configuredIpList = $settings->getIpList();
        $configuredIps = '' !== $configuredIpList ? explode(',', $configuredIpList) : [];
        $envConfiguredIps = '' !== $this->envIpList ? explode(',', $this->envIpList) : [];

        return array_merge($envConfiguredIps, $configuredIps);
    }

    private function isRouteMaintenance($routeName): bool
    {
        return $routeName === 'dedi_maintenance_shop_maintenance';
    }

    private function isMaintenance(RequestEvent $event, string $routeName): bool
    {
        /** @var SettingsInterface $settings */
        $settings = $this->settingsProvider->provide();

        if (!$this->envMaintenanceActive || !$settings->isActive()) {
            return false;
        }

        if ($event->getRequestType() !== HttpKernelInterface::MAIN_REQUEST
        ) {
            return false;
        }

        $request = $event->getRequest();

        if (
            $routeName === '_wdt' ||
            false !== mb_strpos($routeName, '_profiler', 1)
        ) {
            return false;
        }

        $requestUri = $request->getRequestUri();

        if (false !== mb_strpos($requestUri, $this->adminPrefix, 1)) {
            /** @var FlashBagInterface $flashBag */
            $flashBag = $request->getSession()->getBag('flashes');
            $flashBag->add('info', $this->translator->trans('dedi_sylius_maintenance.maintenance_is_active'));

            return false;
        }

        $ipList = $this->getIpList();
        if (count($ipList) !== 0 && in_array($request->getClientIp(), $ipList, true)) {
            return false;
        }

        return true;
    }
}
