<?php

declare(strict_types=1);

namespace Dedi\SyliusMailAdminPlugin\Renderer\Adapter;

use Dedi\SyliusMailAdminPlugin\Event\MailPreRenderEvent;
use Sylius\Bundle\MailerBundle\Renderer\Adapter\EmailTwigAdapter as BaseEmailTwigAdapter;
use Sylius\Component\Mailer\Event\EmailRenderEvent;
use Sylius\Component\Mailer\Model\EmailInterface;
use Sylius\Component\Mailer\Renderer\RenderedEmail;
use Symfony\Contracts\EventDispatcher\EventDispatcherInterface;
use Twig\Environment;
use Twig\Error\LoaderError;
use Twig\Error\RuntimeError;
use Twig\Error\SyntaxError;
use Twig\Loader\ArrayLoader;

class EmailTwigAdapter extends BaseEmailTwigAdapter
{
    public function __construct(
        Environment $twig,
        private readonly string $layoutTemplate,
        ?EventDispatcherInterface $dispatcher = null,
    ) {
        parent::__construct($twig, $dispatcher);
    }

    /**
     * @inheritdoc
     */
    public function render(EmailInterface $email, array $data = []): RenderedEmail
    {
        if (null !== $this->dispatcher) {
            /** @var MailPreRenderEvent $event */
            $event = $this->dispatcher->dispatch(new MailPreRenderEvent($email, $data));

            $email = $event->getEmail();
            $data = $event->getData();
        }

        $renderedEmail = $this->getRenderedEmail($email, $data);

        $event = new EmailRenderEvent($renderedEmail);
        if ($this->dispatcher !== null) {
            /** @var EmailRenderEvent $event */
            $event = $this->dispatcher->dispatch($event);
        }

        return $event->getRenderedEmail();
    }

    private function getRenderedEmail(EmailInterface $email, array $data): RenderedEmail
    {
        if (null !== $email->getTemplate()) {
            return $this->provideEmailWithTemplate($email, $data);
        }

        return $this->provideEmailWithoutTemplate($email, $data);
    }

    /**
     * @psalm-suppress InternalMethod
     */
    private function provideEmailWithTemplate(EmailInterface $email, array $data): RenderedEmail
    {
        $data = $this->twig->mergeGlobals($data);

        $template = $this->twig->load((string) $email->getTemplate());

        $subject = $template->renderBlock('subject', $data);
        $body = $template->renderBlock('body', $data);

        return new RenderedEmail($subject, $body);
    }

    /**
     * @throws LoaderError
     * @throws RuntimeError
     * @throws SyntaxError
     */
    private function provideEmailWithoutTemplate(EmailInterface $email, array $data): RenderedEmail
    {
        $twig = new Environment(new ArrayLoader([]), [
            'autoescape' => false,
        ]);

        $subjectTemplate = $twig->createTemplate((string) $email->getSubject());
        $bodyTemplate = $twig->createTemplate((string) $email->getContent());

        $template = $this->twig->load($this->layoutTemplate);

        $subject = $template->renderBlock('subject', array_merge($data, ['subject' => $subjectTemplate->render($data)]));
        $body = $template->renderBlock('body', array_merge($data, ['body' => $bodyTemplate->render($data)]));

        return new RenderedEmail($subject, $body);
    }
}
