<?php

declare(strict_types=1);

namespace Dedi\SyliusMailAdminPlugin\DependencyInjection\Compiler;

use Dedi\SyliusMailAdminPlugin\Utils\ArrayMerge;
use RuntimeException;
use Symfony\Component\DependencyInjection\Compiler\CompilerPassInterface;
use Symfony\Component\DependencyInjection\ContainerBuilder;
use Symfony\Component\DependencyInjection\Reference;
use Symfony\Component\Yaml\Yaml;

class MailConfigurationPass implements CompilerPassInterface
{
    public function process(ContainerBuilder $container): void
    {
        $this->registerMailVariables($container);
        $this->registerMailPreRenderListeners($container);
    }

    private function registerMailVariables(ContainerBuilder $container): void
    {
        if (!$container->has('dedi_sylius_mail_admin_plugin.form_registry.mail_configuration')) {
            return;
        }

        $mailConfigurations = $container->getExtensionConfig('dedi_sylius_mail_admin');
        $mailVariableRegistry = $container->getDefinition('dedi_sylius_mail_admin_plugin.form_registry.mail_configuration');

        $defaultMailConfiguration = [
            [
                'mails' => Yaml::parseFile(__DIR__ . '/../../Resources/config/default_mail_variables.yaml'),
            ],
        ];
        $mailConfigurations = ArrayMerge::arrayMergeRecursiveDistinct($defaultMailConfiguration, $mailConfigurations);

        if (count($mailConfigurations) > 0 && isset($mailConfigurations[0]['mails'])) {
            foreach ($mailConfigurations[0]['mails'] as $mailCode => $mailConfiguration) {
                $mailVariableRegistry->addMethodCall('addMailConfiguration', [$mailCode, $mailConfiguration['variables']]);
            }
        }
    }

    private function registerMailPreRenderListeners(ContainerBuilder $container): void
    {
        $taggedServices = $container->findTaggedServiceIds('dedi_mail_admin.mail_variables_handler');
        $definition = $container->getDefinition('dedi_sylius_mail_admin.listener.pre_render_listener');

        foreach ($taggedServices as $id => $tags) {
            foreach ($tags as $tag) {
                if (!array_key_exists('mail', $tag)) {
                    throw new RuntimeException(sprintf('Tag is missing "mail" key on service "%s"', $id));
                }

                $definition->addMethodCall('addMailPreRenderListener', [
                    $tag['mail'],
                    new Reference($id),
                ]);
            }
        }
    }
}
