<?php

declare(strict_types=1);

namespace Dedi\SyliusMailAdminPlugin\Checker;

use Dedi\SyliusMailAdminPlugin\Exception\MailConfigurationNotFoundException;
use Dedi\SyliusMailAdminPlugin\Exception\UnexpectedMailVariablesException;
use Dedi\SyliusMailAdminPlugin\Registry\MailConfigurationRegistryInterface;
use Dedi\SyliusMailAdminPlugin\Registry\MailVariable;
use Symfony\Contracts\Translation\TranslatorInterface;

class MailVariablesValidator implements MailVariablesValidatorInterface
{
    public function __construct(
        private readonly MailConfigurationRegistryInterface $mailConfigurationRegistry,
        private readonly TranslatorInterface $translator,
    ) {
    }

    /**
     * @throws UnexpectedMailVariablesException
     */
    public function validateMailVariables(string $mailCode, string $content): void
    {
        try {
            $mailConfiguration = $this->mailConfigurationRegistry->getConfigurationByMailCode($mailCode);
        } catch (MailConfigurationNotFoundException $e) {
            return;
        }

        $availableVariables = array_map(static fn (MailVariable $variable) => $variable->getCode(), $mailConfiguration->getVariables());

        preg_match_all('/{{\s*(?<variables>[^{}\s]*)\s*}}/', $content, $matches);

        if (\count($matches['variables']) === 0) {
            return;
        }

        $submittedVariables = $matches['variables'];
        $unexpectedVariables = array_diff($submittedVariables, $availableVariables);

        if (count($unexpectedVariables) > 0) {
            throw $this->buildException($unexpectedVariables);
        }
    }

    private function buildException(array $unexpectedVariables): UnexpectedMailVariablesException
    {
        return new UnexpectedMailVariablesException(
            $this->translator->trans(
                'dedi_sylius_mail_admin_plugin.unexpected_variables',
                ['%variables%' => implode(', ', $unexpectedVariables)],
                'validators',
            ),
        );
    }
}
