<?php

declare(strict_types=1);

namespace Tests\Dedi\SyliusContactPlugin\Behat\Context\Ui\Admin;

use Behat\Behat\Context\Context;
use Dedi\SyliusContactPlugin\Entity\ContactFormSubjectInterface;
use Sylius\Behat\Service\SharedStorageInterface;
use Tests\Dedi\SyliusContactPlugin\Behat\Page\Admin\Request\CreatePageInterface;
use Tests\Dedi\SyliusContactPlugin\Behat\Page\Admin\Request\IndexPageInterface;
use Tests\Dedi\SyliusContactPlugin\Behat\Page\Admin\Request\UpdatePageInterface;
use Webmozart\Assert\Assert;

final class ManagingSubjectsContext implements Context
{
    private SharedStorageInterface $sharedStorage;

    private CreatePageInterface $createPage;

    private IndexPageInterface $indexPage;

    private UpdatePageInterface $updatePage;

    public function __construct(
        SharedStorageInterface $sharedStorage,
        CreatePageInterface $createPage,
        IndexPageInterface $indexPage,
        UpdatePageInterface $updatePage,
    ) {
        $this->sharedStorage = $sharedStorage;
        $this->createPage = $createPage;
        $this->indexPage = $indexPage;
        $this->updatePage = $updatePage;
    }

    /**
     * @Given I want to create a new subject
     */
    public function iWantToCreateANewSubject(): void
    {
        $this->createPage->open();
    }

    /**
     * @When I name it :label in :language
     * @When I change its label to :label in :language
     */
    public function iNameItIn($label, $language): void
    {
        $this->createPage->specifyLabelIn($label, $language);
    }

    /**
     * @When I specify its recipient's email as :email in :language
     * @When I change its recipient's email to :email in :language
     */
    public function iSpecifyItsRecipientsEmailAs($email, $language): void
    {
        $this->createPage->specifyRecipientEmailIn($email, $language);
    }

    /**
     * @When I specify its rank to :rank
     * @When I change its rank to :rank
     */
    public function iSpecifyItsRankTo($rank): void
    {
        $this->createPage->specifySubjectRank($rank);
    }

    /**
     * @When I add it
     */
    public function iAddIt(): void
    {
        $this->createPage->create();
    }

    /**
     * @Then the subject :label should appear in the registry
     */
    public function theSubjectShouldAppearInTheRegistry(string $label): void
    {
        $this->indexPage->open();
        $this->iShouldSeeTheSubjectInTheList($label);
    }

    /**
     * @When I want to browse subjects
     */
    public function iWantToBrowseSubjects()
    {
        $this->indexPage->open();
    }

    /**
     * @Then I should see :count subjects in the list
     */
    public function iShouldSeeSubjectsInTheList(int $count)
    {
        Assert::eq($this->indexPage->countItems(), $count);
    }

    /**
     * @Then I should see the subject :label in the list
     */
    public function iShouldSeeTheSubjectInTheList(string $label)
    {
        Assert::true($this->indexPage->isSingleResourceOnPage(['label' => $label]));
    }

    /**
     * @When /^I delete (this subject)$/
     */
    public function iDeleteThisSubject(ContactFormSubjectInterface $subject)
    {
        $this->indexPage->open();
        $this->indexPage->deleteResourceOnPage(['label' => $subject->getLabel()]);
    }

    /**
     * @Then /^(this subject) should no longer exist in the registry$/
     */
    public function thisSubjectShouldNoLongerExistInTheRegistry(ContactFormSubjectInterface $subject)
    {
        Assert::false($this->indexPage->isSingleResourceOnPage(['label' => $subject->getLabel()]));
    }

    /**
     * @When /^I want to edit (this subject)$/
     */
    public function iWantToEditThisSubject(ContactFormSubjectInterface $subject)
    {
        $this->updatePage->open(['id' => $subject->getId()]);
    }

    /**
     * @When I disable it
     */
    public function iDisableIt()
    {
        $this->updatePage->disable();
    }

    /**
     * @When I save my changes
     */
    public function iSaveMyChanges()
    {
        $this->updatePage->saveChanges();
    }

    /**
     * @Then /^(this subject) should have the label "([^"]+)" (in the "([^"]+)" locale)$/
     */
    public function thisSubjectShouldHaveTheLabel(ContactFormSubjectInterface $subject, $label, $language)
    {
        $this->updatePage->open(['id' => $subject->getId()]);

        Assert::eq($this->updatePage->getLabel($language), $label);
    }

    /**
     * @Then /^(this subject) should have the recipient email "([^"]+)" (in the "([^"]+)" locale)$/
     */
    public function thisSubjectShouldHaveTheRecipientEmail(ContactFormSubjectInterface $subject, $email, $language)
    {
        $this->updatePage->open(['id' => $subject->getId()]);

        Assert::eq($this->updatePage->getRecipientEmail($language), $email);
    }

    /**
     * @Then /^(this subject) should have the rank ([^"]+)$/
     */
    public function thisSubjectShouldHaveTheRank(ContactFormSubjectInterface $subject, $rank)
    {
        $this->updatePage->open(['id' => $subject->getId()]);

        Assert::eq($this->updatePage->getRank(), $rank);
    }

    /**
     * @Then /^(this subject) should be disabled$/
     */
    public function thisSubjectShouldBeDisabled(ContactFormSubjectInterface $subject)
    {
        $this->updatePage->open(['id' => $subject->getId()]);

        Assert::false($this->updatePage->isEnabled());
    }
}
