<?php

declare(strict_types=1);

namespace Dedi\SyliusContactPlugin\Form;

use Dedi\SyliusContactPlugin\Entity\ContactFormSubject;
use Dedi\SyliusContactPlugin\Repository\ContactFormSubjectRepositoryInterface;
use EWZ\Bundle\RecaptchaBundle\Form\Type\EWZRecaptchaType;
use EWZ\Bundle\RecaptchaBundle\Validator\Constraints\IsTrue as RecaptchaTrue;
use Sylius\Component\Locale\Context\LocaleContextInterface;
use Symfony\Bridge\Doctrine\Form\Type\EntityType;
use Symfony\Component\Form\AbstractType;
use Symfony\Component\Form\Extension\Core\Type\EmailType;
use Symfony\Component\Form\Extension\Core\Type\TextareaType;
use Symfony\Component\Form\Extension\Core\Type\TextType;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\OptionsResolver\OptionsResolver;
use Symfony\Component\Validator\Constraints\Email;
use Symfony\Component\Validator\Constraints\NotBlank;
use Symfony\Component\Validator\Constraints\Regex;

class ContactFormMessageType extends AbstractType
{
    public function __construct(
        private readonly LocaleContextInterface $localeContext,
        private readonly ContactFormSubjectRepositoryInterface $contactFormSubjectRepository,
        private readonly bool $invisibleCaptcha,
        private readonly string $phonePattern,
    ) {
    }

    public function buildForm(FormBuilderInterface $builder, array $options): void
    {
        if ($this->contactFormSubjectRepository->hasMultipleContactFormMessage()) {
            $builder->add('subject', EntityType::class, [
                'class' => ContactFormSubject::class,
                'label' => 'dedi_sylius_contact_form.ui.form.subject',
                'choice_label' => 'label',
                'choice_value' => 'id',
                'query_builder' => fn (ContactFormSubjectRepositoryInterface $er) => $er
                    ->createTranslationBasedQueryBuilder($this->localeContext->getLocaleCode())
                    ->andWhere('o.active = :status')
                    ->orderBy('o.subjectRank')
                    ->setParameter('status', 1),
                'constraints' => [
                    new NotBlank(['groups' => ['default', 'sylius']]),
                ],
            ]);
        }

        $builder
            ->add('name', TextType::class, [
                'label' => 'dedi_sylius_contact_form.ui.form.name',
                'constraints' => [
                    new NotBlank(['groups' => ['default', 'sylius']]),
                ],
            ])
            ->add('email', EmailType::class, [
                'label' => 'dedi_sylius_contact_form.ui.form.email',
                'constraints' => [
                    new NotBlank(['groups' => ['default', 'sylius']]),
                    new Email(['groups' => ['default', 'sylius']]),
                ],
            ])
            ->add('message', TextareaType::class, [
                'label' => 'dedi_sylius_contact_form.ui.form.message',
                'constraints' => [
                    new NotBlank(['groups' => ['default', 'sylius']]),
                ],
            ])
            ->add('phone', TextType::class, [
                'label' => 'dedi_sylius_contact_form.ui.form.phone',
                'required' => false,
                'attr' => [
                    'pattern' => $this->phonePattern,
                ],
                'constraints' => [
                    new Regex([
                        'pattern' => sprintf('/%s/', $this->phonePattern),
                        'groups' => ['default', 'sylius'],
                    ]),
                ],
            ])
            ->add('attachments', AttachmentCollectionType::class)
            ->add('captcha', EWZRecaptchaType::class, [
                'mapped' => false,
                'label' => $this->invisibleCaptcha ? false : 'dedi_sylius_contact_form.ui.form.captcha',
                'attr' => [
                    'options' => [
                        'theme' => 'light',
                        'type' => 'image',
                        'size' => $this->invisibleCaptcha ? 'invisible' : 'normal',
                        'defer' => true,
                        'async' => true,
                        'bind' => 'recaptcha-submit',
                    ],
                ],
                'constraints' => [
                    new RecaptchaTrue(['groups' => ['default', 'sylius']]),
                ],
            ])
        ;
    }

    public function configureOptions(OptionsResolver $resolver): void
    {
        $resolver
            ->setDefaults([
                'validation_groups' => ['default', 'sylius'],
            ])
        ;
    }
}
