<?php

declare(strict_types=1);

namespace Dedi\SyliusContactPlugin\Controller;

use Dedi\SyliusContactPlugin\Entity\ContactFormSubjectInterface;
use Dedi\SyliusContactPlugin\Factory\ContactFormMessageFactoryInterface;
use Dedi\SyliusContactPlugin\Factory\ContactFormSubjectFactoryInterface;
use Dedi\SyliusContactPlugin\Form\Shop\ContactFormMessageType;
use Dedi\SyliusContactPlugin\Mailer\ContactRequestMailerInterface;
use Dedi\SyliusContactPlugin\Mailer\CustomerContactRequestMailerInterface;
use Dedi\SyliusContactPlugin\Repository\ContactFormSubjectRepositoryInterface;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Contracts\Translation\TranslatorInterface;

class ContactFormController extends AbstractController
{
    public function __construct(
        private readonly ContactFormSubjectRepositoryInterface $contactFormSubjectRepository,
        private readonly ContactFormMessageFactoryInterface $messageFactory,
        private readonly ContactFormSubjectFactoryInterface $subjectFactory,
        private readonly EntityManagerInterface $entityManager,
        private readonly TranslatorInterface $translator,
        private readonly ContactRequestMailerInterface $contactRequestMailer,
        private readonly CustomerContactRequestMailerInterface $customerContactRequestMailer,
    ) {
    }

    public function requestAction(Request $request): Response
    {
        $form = $this->createForm(ContactFormMessageType::class);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $data = $form->getData();

            /** @var ContactFormSubjectInterface $subject */
            $subject = $data['subject'] ?? $this->getFallbackSubject();
            $contactMessage = $this->messageFactory->createNewFromSubjectDataAndRequest(
                $subject,
                $data,
                $request,
            );

            $this->entityManager->persist($contactMessage);
            $this->entityManager->flush();

            $this->contactRequestMailer->send($contactMessage);
            $this->customerContactRequestMailer->send($contactMessage);

            $this->addFlash('success', $this->translator->trans(
                $this->getSyliusAttribute(
                    $request,
                    'success_flash',
                    'dedi_sylius_contact_form.request_success',
                ),
            ));

            // create a new form in order for the form not to be pre-filled
            $form = $this->createForm(ContactFormMessageType::class);
        }

        $template = $this->getSyliusAttribute(
            $request,
            'template',
            '@SyliusShop/contact/contact_request.html.twig',
        );

        return $this->render($template, ['form' => $form->createView()]);
    }

    private function getFallbackSubject(): ContactFormSubjectInterface
    {
        /** @var ?ContactFormSubjectInterface $subject */
        $subject = $this->contactFormSubjectRepository->findOneBy(['active' => true]);
        if (null !== $subject) {
            return $subject;
        }

        $defaultSubject = $this->subjectFactory->createDefault();
        $this->contactFormSubjectRepository->add($defaultSubject);

        return $defaultSubject;
    }

    private function getSyliusAttribute(Request $request, string $attributeName, ?string $default): ?string
    {
        $attributes = $request->attributes->get('_sylius');

        return $attributes[$attributeName] ?? $default;
    }
}
